/*
 * Copyright 2010-2015 http://beders.github.com/Resty
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.rest.web;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URLConnection;

/**
 * 
 * Represents a chunked, streaming content. Takes the data from InputStream and
 * sends it.
 * 
 * @author ashish
 */
public class ChunkedContent extends AbstractContent {

	private static final int BUFFER_SIZE = 4096;
	private int chunkSize = BUFFER_SIZE;
	private final InputStream in;
	private final String mime;

	/**
	 * Instantiates a new chunked content. If streaming mode is supported by
	 * the provided connection ({@link HttpURLConnection}), Resty will use it in streaming mode.
	 * 
	 * @param mime
	 *            the mime
	 * @param in
	 *            the in
	 * @param chunkSize
	 *            the chunk size. If chunk size is negative or 0, Resty will use
	 *            its default buffer size as chunk size.
	 */
	public ChunkedContent(String mime, InputStream in, int chunkSize) {
		this.in = in;
		this.mime = mime;
		if (chunkSize > 0) {
			this.chunkSize = chunkSize;
		}
	}

	@Override
	public void writeHeader(OutputStream os) throws IOException {
		os.write(ascii("Content-Type: " + mime + "\r\n"));
		os.write(ascii("Transfer-Encoding: chunked\r\n"));
	}

	@Override
	public void writeContent(OutputStream os) throws IOException {
		byte[] buffer = new byte[BUFFER_SIZE];
		int len;
		while ((len = in.read(buffer)) != -1) {
			os.write(buffer, 0, len);
		}
	}

	@Override
	protected void addContent(URLConnection con) throws IOException {
		if(con instanceof HttpURLConnection){
			((HttpURLConnection) con).setChunkedStreamingMode(chunkSize);
		}
		con.setDoOutput(true);
		con.addRequestProperty("Content-Type", mime);
		con.addRequestProperty("Transfer-Encoding", "chunked");
		OutputStream os = con.getOutputStream();
		writeContent(os);
		os.close();
	}

}
