/*
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.jsonpath.parser;

/**
 * An implementation of interface CharStream, where the stream is assumed to contain only ASCII characters (without
 * unicode processing) adapted from the generated class.
 */

public class SimpleCharStream implements CharStream {

	private static final int DEFAULT_BUFFER_SIZE = Integer
			.parseInt(System.getProperty("ej.rest.jsonquerypath.buffersize", "128"));
	private static final int EXPAND_BUFFER_SIZE = Integer
			.parseInt(System.getProperty("ej.rest.jsonquerypath.expandbuffersize", "32"));

	int bufsize;
	int available;
	int tokenBegin;
	/** Position in buffer. */
	public int bufpos = -1;
	protected int bufline[];
	protected int bufcolumn[];

	protected int column = 0;
	protected int line = 1;

	protected boolean prevCharIsCR = false;
	protected boolean prevCharIsLF = false;

	protected java.io.Reader inputStream;

	protected char[] buffer;
	protected int maxNextCharInd = 0;
	protected int inBuf = 0;
	protected int tabSize = 1;
	protected boolean trackLineColumn = true;

	@Override
	public void setTabSize(int i) {
		this.tabSize = i;
	}

	@Override
	public int getTabSize() {
		return this.tabSize;
	}

	protected void ExpandBuff(boolean wrapAround) {
		int bufsize = this.bufsize;
		int newBufsize = bufsize + EXPAND_BUFFER_SIZE;
		char[] newbuffer = new char[newBufsize];
		int newbufline[] = new int[newBufsize];
		int newbufcolumn[] = new int[newBufsize];

		try {
			char[] buffer = this.buffer;
			int tokenBegin = this.tokenBegin;
			int[] bufline = this.bufline;
			int[] bufcolumn = this.bufcolumn;
			if (wrapAround) {
				System.arraycopy(buffer, tokenBegin, newbuffer, 0, bufsize - tokenBegin);
				int bufpos = this.bufpos;
				System.arraycopy(buffer, 0, newbuffer, bufsize - tokenBegin, bufpos);
				this.buffer = newbuffer;

				System.arraycopy(bufline, tokenBegin, newbufline, 0, bufsize - tokenBegin);
				System.arraycopy(bufline, 0, newbufline, bufsize - tokenBegin, bufpos);
				this.bufline = newbufline;

				System.arraycopy(bufcolumn, tokenBegin, newbufcolumn, 0, bufsize - tokenBegin);
				System.arraycopy(bufcolumn, 0, newbufcolumn, bufsize - tokenBegin, bufpos);
				this.bufcolumn = newbufcolumn;

				this.maxNextCharInd = (this.bufpos += (bufsize - tokenBegin));
			} else {
				System.arraycopy(buffer, tokenBegin, newbuffer, 0, bufsize - tokenBegin);
				this.buffer = newbuffer;

				System.arraycopy(bufline, tokenBegin, newbufline, 0, bufsize - tokenBegin);
				this.bufline = newbufline;

				System.arraycopy(bufcolumn, tokenBegin, newbufcolumn, 0, bufsize - tokenBegin);
				this.bufcolumn = newbufcolumn;

				this.maxNextCharInd = (this.bufpos -= tokenBegin);
			}
		} catch (Throwable t) {
			throw new Error(t.getMessage());
		}

		this.bufsize += 2048;
		this.available = bufsize;
		this.tokenBegin = 0;
	}

	protected void FillBuff() throws java.io.IOException {
		int tokenBegin = this.tokenBegin;
		int available = this.available;
		if (this.maxNextCharInd == available) {
			if (available == this.bufsize) {
				if (tokenBegin > EXPAND_BUFFER_SIZE) {
					this.bufpos = this.maxNextCharInd = 0;
					this.available = tokenBegin;
				} else if (tokenBegin < 0) {
					this.bufpos = this.maxNextCharInd = 0;
				} else {
					ExpandBuff(false);
				}
			} else if (available > tokenBegin) {
				this.available = this.bufsize;
			} else if ((tokenBegin - available) < EXPAND_BUFFER_SIZE) {
				ExpandBuff(true);
			} else {
				this.available = tokenBegin;
			}
		}
		available = this.available;

		int i;
		try {
			if ((i = this.inputStream.read(this.buffer, this.maxNextCharInd, available - this.maxNextCharInd)) == -1) {
				this.inputStream.close();
				throw new java.io.IOException();
			} else {
				this.maxNextCharInd += i;
			}
			return;
		} catch (java.io.IOException e) {
			--this.bufpos;
			backup(0);
			if (tokenBegin == -1) {
				this.tokenBegin = this.bufpos;
			}
			throw e;
		}
	}

	/** Start. */
	@Override
	public char BeginToken() throws java.io.IOException {
		this.tokenBegin = -1;
		char c = readChar();
		this.tokenBegin = this.bufpos;

		return c;
	}

	protected void UpdateLineColumn(char c) {
		this.column++;

		if (this.prevCharIsLF) {
			this.prevCharIsLF = false;
			this.line += (this.column = 1);
		} else if (this.prevCharIsCR) {
			this.prevCharIsCR = false;
			if (c == '\n') {
				this.prevCharIsLF = true;
			} else {
				this.line += (this.column = 1);
			}
		}

		switch (c) {
		case '\r':
			this.prevCharIsCR = true;
			break;
		case '\n':
			this.prevCharIsLF = true;
			break;
		case '\t':
			this.column--;
			this.column += (this.tabSize - (this.column % this.tabSize));
			break;
		default:
			break;
		}

		this.bufline[this.bufpos] = this.line;
		this.bufcolumn[this.bufpos] = this.column;
	}

	/** Read a character. */
	@Override
	public char readChar() throws java.io.IOException {
		if (this.inBuf > 0) {
			--this.inBuf;

			if (++this.bufpos == this.bufsize) {
				this.bufpos = 0;
			}

			return this.buffer[this.bufpos];
		}

		if (++this.bufpos >= this.maxNextCharInd) {
			FillBuff();
		}

		char c = this.buffer[this.bufpos];

		UpdateLineColumn(c);
		return c;
	}

	@Override
	@Deprecated
	/**
	 * @deprecated
	 * @see #getEndColumn
	 */
	public int getColumn() {
		return this.bufcolumn[this.bufpos];
	}

	@Override
	@Deprecated
	/**
	 * @deprecated
	 * @see #getEndLine
	 */
	public int getLine() {
		return this.bufline[this.bufpos];
	}

	/** Get token end column number. */
	@Override
	public int getEndColumn() {
		return this.bufcolumn[this.bufpos];
	}

	/** Get token end line number. */
	@Override
	public int getEndLine() {
		return this.bufline[this.bufpos];
	}

	/** Get token beginning column number. */
	@Override
	public int getBeginColumn() {
		return this.bufcolumn[this.tokenBegin];
	}

	/** Get token beginning line number. */
	@Override
	public int getBeginLine() {
		return this.bufline[this.tokenBegin];
	}

	/** Backup a number of characters. */
	@Override
	public void backup(int amount) {

		this.inBuf += amount;
		if ((this.bufpos -= amount) < 0) {
			this.bufpos += this.bufsize;
		}
	}

	/** Constructor. */
	public SimpleCharStream(java.io.Reader dstream, int startline, int startcolumn, int buffersize) {
		this.inputStream = dstream;
		this.line = startline;
		this.column = startcolumn - 1;

		this.available = this.bufsize = buffersize;
		this.buffer = new char[buffersize];
		this.bufline = new int[buffersize];
		this.bufcolumn = new int[buffersize];
	}

	/** Constructor. */
	public SimpleCharStream(java.io.Reader dstream, int startline, int startcolumn) {
		this(dstream, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.Reader dstream) {
		this(dstream, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.Reader dstream, int startline, int startcolumn, int buffersize) {
		this.inputStream = dstream;
		this.line = startline;
		this.column = startcolumn - 1;

		if (this.buffer == null || buffersize != this.buffer.length) {
			this.available = this.bufsize = buffersize;
			this.buffer = new char[buffersize];
			this.bufline = new int[buffersize];
			this.bufcolumn = new int[buffersize];
		}
		this.prevCharIsLF = this.prevCharIsCR = false;
		this.tokenBegin = this.inBuf = this.maxNextCharInd = 0;
		this.bufpos = -1;
	}

	/** Reinitialise. */
	public void ReInit(java.io.Reader dstream, int startline, int startcolumn) {
		ReInit(dstream, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.Reader dstream) {
		ReInit(dstream, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream, String encoding, int startline, int startcolumn,
			int buffersize) throws java.io.UnsupportedEncodingException {
		this(encoding == null ? new java.io.InputStreamReader(dstream)
				: new java.io.InputStreamReader(dstream, encoding), startline, startcolumn, buffersize);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream, int startline, int startcolumn, int buffersize) {
		this(new java.io.InputStreamReader(dstream), startline, startcolumn, buffersize);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream, String encoding, int startline, int startcolumn)
			throws java.io.UnsupportedEncodingException {
		this(dstream, encoding, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream, int startline, int startcolumn) {
		this(dstream, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream, String encoding) throws java.io.UnsupportedEncodingException {
		this(dstream, encoding, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Constructor. */
	public SimpleCharStream(java.io.InputStream dstream) {
		this(dstream, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream, String encoding, int startline, int startcolumn, int buffersize)
			throws java.io.UnsupportedEncodingException {
		ReInit(encoding == null ? new java.io.InputStreamReader(dstream)
				: new java.io.InputStreamReader(dstream, encoding), startline, startcolumn, buffersize);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream, int startline, int startcolumn, int buffersize) {
		ReInit(new java.io.InputStreamReader(dstream), startline, startcolumn, buffersize);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream, String encoding) throws java.io.UnsupportedEncodingException {
		ReInit(dstream, encoding, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream) {
		ReInit(dstream, 1, 1, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream, String encoding, int startline, int startcolumn)
			throws java.io.UnsupportedEncodingException {
		ReInit(dstream, encoding, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Reinitialise. */
	public void ReInit(java.io.InputStream dstream, int startline, int startcolumn) {
		ReInit(dstream, startline, startcolumn, DEFAULT_BUFFER_SIZE);
	}

	/** Get token literal value. */
	@Override
	public String GetImage() {
		if (this.bufpos >= this.tokenBegin) {
			return new String(this.buffer, this.tokenBegin, this.bufpos - this.tokenBegin + 1);
		} else {
			return new String(this.buffer, this.tokenBegin, this.bufsize - this.tokenBegin)
					+ new String(this.buffer, 0, this.bufpos + 1);
		}
	}

	/** Get the suffix. */
	@Override
	public char[] GetSuffix(int len) {
		char[] ret = new char[len];

		if ((this.bufpos + 1) >= len) {
			System.arraycopy(this.buffer, this.bufpos - len + 1, ret, 0, len);
		} else {
			System.arraycopy(this.buffer, this.bufsize - (len - this.bufpos - 1), ret, 0, len - this.bufpos - 1);
			System.arraycopy(this.buffer, 0, ret, len - this.bufpos - 1, this.bufpos + 1);
		}

		return ret;
	}

	/** Reset buffer when finished. */
	@Override
	public void Done() {
		this.buffer = null;
		this.bufline = null;
		this.bufcolumn = null;
	}

	/**
	 * Method to adjust line and column numbers for the start of a token.
	 */
	public void adjustBeginLineColumn(int newLine, int newCol) {
		int tokenBegin = this.tokenBegin;
		int start = tokenBegin;
		int len;

		int bufsize = this.bufsize;
		int bufpos = this.bufpos;
		int inBuf = this.inBuf;
		if (bufpos >= tokenBegin) {
			len = bufpos - tokenBegin + inBuf + 1;
		} else {
			len = bufsize - tokenBegin + bufpos + 1 + inBuf;
		}

		int i = 0, j = 0, k = 0;
		int nextColDiff = 0, columnDiff = 0;

		int[] bufline = this.bufline;
		int[] bufcolumn = this.bufcolumn;
		while (i < len && bufline[j = start % bufsize] == bufline[k = ++start % bufsize]) {
			bufline[j] = newLine;
			nextColDiff = columnDiff + bufcolumn[k] - bufcolumn[j];
			bufcolumn[j] = newCol + columnDiff;
			columnDiff = nextColDiff;
			i++;
		}

		if (i < len) {
			bufline[j] = newLine++;
			bufcolumn[j] = newCol + columnDiff;

			while (i++ < len) {
				if (bufline[j = start % bufsize] != bufline[++start % bufsize]) {
					bufline[j] = newLine++;
				} else {
					bufline[j] = newLine;
				}
			}
		}

		this.line = bufline[j];
		this.column = bufcolumn[j];
	}

	@Override
	public boolean getTrackLineColumn() {
		return this.trackLineColumn;
	}

	@Override
	public void setTrackLineColumn(boolean tlc) {
		this.trackLineColumn = tlc;
	}
}
