/*
 * Java
 *
 * Copyright 2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.tool.addon;

import java.io.File;
import java.io.FilenameFilter;
import java.util.List;

/**
 * An addon processor is a tool that generates files (Java source files, resource files,...) for a specific project.
 * Given project information and deltas, it can generates new files for this project.
 *
 * <p>
 * Since there is no guarantee that processor caller will create an instance per project or a unique instance of a
 * processor for all projects, processors must be stateless, thus only rely on given information and deltas.
 *
 * @see Delta
 * @see Marker
 * @see JavaProject
 */
public interface AddonProcessor {

	/**
	 * An enumeration of possible kind of processor execution.
	 */
	enum ExecutionKind {
		/**
		 * Full execution means processor must generate all files; preceding generated files have been cleared.
		 */
		Full,
		/**
		 * Incremental execution means the processor can generate a subset of files. Preceding generated files are still
		 * in the given output folder.
		 */
		Incremental
	}

	/**
	 * The name of the Ivy configuration to use to declare addon processors.
	 */
	String ADDON_PROCESSOR_IVY_CONF = "addon-processor"; //$NON-NLS-1$
	/**
	 * Addon Processor file extension.
	 */
	String ADDON_PROCESSOR_FILE_EXTENSION = ".adp"; //$NON-NLS-1$

	/**
	 * Gets the ID of this processor.
	 *
	 * <p>
	 * IDs are used to create output source folders. Thus, they must not contain space or special characters.
	 *
	 * @return the id of this processor.
	 */
	String getId();

	/**
	 * Gets the preferred output folder kind of this processor.
	 *
	 * <p>
	 * Note that {@link FolderKind#Project} is not a valid preferred output folder. If this value is returned, an
	 * exception will be thrown, thus processor will not be run.
	 *
	 * <p>
	 * In addition, if this method returns {@code null}, executors will automatically use Java main source folder as
	 * preferred output kind.
	 *
	 * @return the preferred output folder kind of this processor.
	 *
	 * @see FolderKind
	 */
	FolderKind getPreferredOutputFolderKind();

	/**
	 * Given project information, gets a filter to evict project deltas that are not needed by this processor.
	 *
	 * @param project
	 *            Java project information the filter can rely on.
	 *
	 * @return this processor delta filter.
	 */
	FilenameFilter getDeltaFilter(JavaProject project);

	/**
	 * Runs this processor. Based on project information and deltas, files can be generated in the given output folder.
	 *
	 * <p>
	 * Note that deltas list cannot be empty nor {@code null}, and all of them match this processor delta filter.
	 * Furthermore, the given output folder respects the semantics of {@link #getPreferredOutputFolderKind()}.
	 *
	 * @param kind
	 *            the kind of execution.
	 * @param project
	 *            Java project information to rely on.
	 * @param deltas
	 *            list of project deltas.
	 * @param outputFolder
	 *            the output folder for generated files.
	 * @return the list of markers
	 */
	List<Marker> execute(ExecutionKind kind, JavaProject project, List<Delta> deltas, File outputFolder);

}
