/*
 * Java
 *
 * Copyright 2014-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.impl;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UTFDataFormatException;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import ej.annotation.Nullable;
import ej.rcommand.RemoteConnection;

public class StreamRemoteConnection implements RemoteConnection {

	private static final int DEFAULT_BUFFER_SIZE = 4096;

	private static final byte TYPE_STRING = 'S';
	private static final byte TYPE_LONG = 'J';
	private static final byte TYPE_BOOLEAN = 'Z';
	private static final byte TYPE_INT = 'I';
	private static final byte TYPE_FLOAT = 'F';
	private static final byte TYPE_DOUBLE = 'D';
	private static final byte TYPE_BYTE_ARRAY = 'A';
	private static final byte TYPE_INPUT_STREAM = 'T';
	private static final byte TYPE_END_OF_COMMAND = 'E';

	private final DataInputStream in;
	private final DataOutputStream out;
	private boolean closed;

	private boolean locked;

	/** Set to true when End Of Command has been read. */
	private boolean eofCommandDetected;

	public StreamRemoteConnection(InputStream is, OutputStream os) {
		this.in = new DataInputStream(is);
		this.out = new DataOutputStream(os);
	}

	private Logger getLogger() {
		Logger logger = Logger.getLogger(StreamRemoteConnection.class.getName());
		assert (logger != null);
		return logger;
	}

	private void logError(IOException e) {
		getLogger().log(Level.SEVERE, e.getMessage(), e);
	}

	@Override
	public void startCommand(String command) {
		takeLock();
		sendString(command);
	}

	@Override
	public void sendString(String s) {
		try {
			this.out.writeByte(TYPE_STRING);
			this.out.writeUTF(s);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendLong(long l) {
		try {
			this.out.writeByte(TYPE_LONG);
			this.out.writeLong(l);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendInt(int i) {
		try {
			this.out.writeByte(TYPE_INT);
			this.out.writeInt(i);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendFloat(float f) {
		try {
			this.out.writeByte(TYPE_FLOAT);
			this.out.writeFloat(f);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendDouble(double d) {
		try {
			this.out.writeByte(TYPE_DOUBLE);
			this.out.writeDouble(d);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendBoolean(boolean b) {
		try {
			this.out.writeByte(TYPE_BOOLEAN);
			this.out.writeBoolean(b);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendByteArray(byte[] a) {
		try {
			this.out.writeByte(TYPE_BYTE_ARRAY);
			this.out.writeInt(a.length);
			this.out.write(a);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendByteArray(byte[] buffer, int offset, int length) {
		try {
			this.out.writeByte(TYPE_BYTE_ARRAY);
			this.out.writeInt(length);
			this.out.write(buffer, offset, length);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void sendByteArrayAsInputStream(InputStream is) {
		try {
			this.out.writeByte(TYPE_BYTE_ARRAY);
			int length = is.available();
			this.out.writeInt(length);
			byte[] buffer = new byte[64];
			int index = 0;
			while (index != length) {
				int count = is.read(buffer);
				this.out.write(buffer, 0, count);
				index += count;
			}
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public void flushCommand() {
		try {
			this.out.writeByte(TYPE_END_OF_COMMAND);
			this.out.flush();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
		releaseLock();
	}

	protected void takeLock() {
		synchronized (this) {
			while (this.locked) {
				try {
					this.wait();
				} catch (InterruptedException e) {
				}
			}
			this.locked = true;
		}
	}

	protected void releaseLock() {
		synchronized (this) {
			this.locked = false;
			notifyAll();
		}
	}

	@Override
	public String readString() throws IOException {
		try {
			checkType(TYPE_STRING);
			return this.in.readUTF();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			return "";
		}
	}

	@Override
	public String readCommand() throws IOException {
		try {
			checkType(TYPE_STRING);
			return this.in.readUTF();
		} catch (EOFException e) {
			// it is fine to receive an eof between two commands. Do not log it
			throw e;
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public long readLong() throws IOException {
		try {
			checkType(TYPE_LONG);
			return this.in.readLong();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public int readInt() throws IOException {
		try {
			checkType(TYPE_INT);
			return this.in.readInt();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public float readFloat() throws IOException {
		try {
			checkType(TYPE_FLOAT);
			return this.in.readFloat();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public double readDouble() throws IOException {
		try {
			checkType(TYPE_DOUBLE);
			return this.in.readDouble();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public boolean readBoolean() throws IOException {
		try {
			checkType(TYPE_BOOLEAN);
			return this.in.readBoolean();
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public byte[] readByteArray() throws IOException {
		try {
			checkType(TYPE_BYTE_ARRAY);
			return readByteArrayIntern();
		} catch (Throwable e) {
			if (!this.closed) {
				getLogger().log(Level.SEVERE, e.getMessage(), e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public InputStream readByteArrayAsInputStream() throws IOException {
		try {
			checkType(TYPE_BYTE_ARRAY);
			return readByteArrayAsInputStreamIntern();
		} catch (Throwable e) {
			if (!this.closed) {
				getLogger().log(Level.SEVERE, e.getMessage(), e);
				throw e;
			}
			throw new EOFException();
		}
	}

	private byte[] readByteArrayIntern() throws IOException {
		int length = this.in.readInt();
		if (length < 0) {
			return new byte[0];
		}

		try {
			byte[] a = new byte[length];
			this.in.readFully(a);
			return a;
		} catch (OutOfMemoryError e) {
			e.printStackTrace();
			return new byte[0];
		}
	}

	private InputStream readByteArrayAsInputStreamIntern() throws IOException {
		int length = this.in.readInt();
		return new LimitedLengthInputStream(this.in, length);
	}

	@Override
	public Object readObject() throws IOException {
		try {
			byte type = this.in.readByte();
			Object param;
			switch (type) {
			case TYPE_BYTE_ARRAY:
				param = readByteArrayAsInputStreamIntern();
				break;
			case TYPE_STRING:
				param = this.in.readUTF();
				break;
			case TYPE_LONG:
				long l = this.in.readLong();
				param = Long.valueOf(l);
				break;
			case TYPE_BOOLEAN:
				boolean b = this.in.readBoolean();
				param = Boolean.valueOf(b);
				break;
			case TYPE_INT:
				int i = this.in.readInt();
				param = Integer.valueOf(i);
				break;
			case TYPE_FLOAT:
				float f = this.in.readFloat();
				param = Float.valueOf(f);
				break;
			case TYPE_DOUBLE:
				double d = this.in.readDouble();
				param = Double.valueOf(d);
				break;
			default:
				throw new IllegalArgumentException("Invalid parameter type: 0x" + Integer.toHexString(type));
			}
			return param;
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public List<Object> readParameters() throws IOException {
		List<Object> params = new ArrayList<>();
		try {
			while (true) {
				byte type = this.in.readByte();
				Object param;
				switch (type) {
				case TYPE_BYTE_ARRAY:
					param = readByteArrayIntern();
					break;
				case TYPE_STRING:
					param = this.in.readUTF();
					break;
				case TYPE_LONG:
					long l = this.in.readLong();
					param = Long.valueOf(l);
					break;
				case TYPE_BOOLEAN:
					boolean b = this.in.readBoolean();
					param = Boolean.valueOf(b);
					break;
				case TYPE_INT:
					int i = this.in.readInt();
					param = Integer.valueOf(i);
					break;
				case TYPE_FLOAT:
					float f = this.in.readFloat();
					param = Float.valueOf(f);
					break;
				case TYPE_DOUBLE:
					double d = this.in.readDouble();
					param = Double.valueOf(d);
					break;
				case TYPE_END_OF_COMMAND:
					this.eofCommandDetected = true;
					return params;
				default:
					throw new InternalError("Invalid parameter type.");
				}
				params.add(param);
			}
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			throw new EOFException();
		}
	}

	@Override
	public void sendParams(List<Object> params) {
		for (Object param : params) {
			if (param.getClass().isArray()) {
				byte[] array = (byte[]) param;
				sendByteArray(array);
			} else if (param instanceof String) {
				String s = (String) param;
				sendString(s);
			} else if (param instanceof Long) {
				Long l = (Long) param;
				sendLong(l.longValue());
			} else if (param instanceof Integer) {
				Integer i = (Integer) param;
				sendInt(i.intValue());
			} else if (param instanceof Float) {
				Float f = (Float) param;
				sendFloat(f.floatValue());
			} else if (param instanceof Double) {
				Double d = (Double) param;
				sendDouble(d.doubleValue());
			} else if (param instanceof Boolean) {
				Boolean b = (Boolean) param;
				sendBoolean(b.booleanValue());
			} else {
				throw new InternalError("Invalid parameter type: " + param.getClass());
			}
		}
	}

	@Override
	public void skipParameters() throws IOException {
		if (this.eofCommandDetected) {
			this.eofCommandDetected = false;
			return;
		}

		boolean errorDetected = false;

		try {
			while (true) {
				byte type = this.in.readByte();
				switch (type) {
				case TYPE_BYTE_ARRAY:
					InputStream stream = readByteArrayAsInputStreamIntern();
					byte[] chuncks = new byte[LimitedLengthInputStream.CHUNCKS_SIZE];
					while (stream.read(chuncks) != -1) {
						// do nothing
						// just consume the input stream
					}
					break;
				case TYPE_STRING:
					try {
						this.in.readUTF();
					} catch (UTFDataFormatException | Error e) {
						if (!errorDetected) {
							getLogger().severe("Invalid UTF string.");
							errorDetected = true;
						}
					}
					break;
				case TYPE_LONG:
					this.in.readLong();
					break;
				case TYPE_FLOAT:
					this.in.readFloat();
					break;
				case TYPE_DOUBLE:
					this.in.readDouble();
					break;
				case TYPE_BOOLEAN:
					this.in.readBoolean();
					break;
				case TYPE_INT:
					this.in.readInt();
					break;
				case TYPE_INPUT_STREAM:
					int chunkSize = this.in.readInt();
					while (chunkSize != 0) {
						while (chunkSize > 0) {
							chunkSize -= this.in.skip(chunkSize);
						}
						chunkSize = this.in.readInt();
					}
					break;
				case TYPE_END_OF_COMMAND:
					if (errorDetected) {
						getLogger().warning("Invalid frame skipped");
					}
					return;
				default:
					if (!errorDetected) {
						getLogger().severe("Invalid parameter type (0x" + Integer.toHexString(type & 0xFF) + ").");
						errorDetected = true;
					}
				}
			}
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
				throw e;
			}
			return;
		}
	}

	@Override
	public void close() {
		this.closed = true;

		try {
			this.out.close();
		} catch (IOException e) {
		}
		try {
			this.in.close();
		} catch (IOException e) {
		}
	}

	/**
	 * Checks that the type of the read parameter is the right one
	 */
	private void checkType(byte expected) throws IOException {
		byte type = this.in.readByte();
		if (type != expected) {
			throw new IllegalArgumentException("Invalid parameter type: 0x" + Integer.toHexString(type) + " expected 0x"
					+ Integer.toHexString(expected));
		}
	}

	@Override
	public void sendInputStream(InputStream is) {
		try {
			this.out.writeByte(TYPE_INPUT_STREAM);
			byte[] buffer = new byte[DEFAULT_BUFFER_SIZE];
			int read = is.read(buffer);
			while (read != -1) {
				this.out.writeInt(read);
				this.out.write(buffer, 0, read);
				this.out.flush();
				read = is.read(buffer);
			}
			this.out.writeInt(0);
		} catch (IOException e) {
			if (!this.closed) {
				logError(e);
			}
		}
	}

	@Override
	public InputStream readInputStream() throws IOException {
		checkType(TYPE_INPUT_STREAM);
		return new ChunkedInputStream(this.in);
	}

	@Override
	public @Nullable OutputStream getOutputStream() {
		return this.out;
	}

	@Override
	public @Nullable InputStream getInputStream() {
		return this.in;
	}

}
