/*
 * Java
 *
 * Copyright 2014-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket.frame;

import java.util.Random;

import ej.util.message.Level;
import ej.websocket.Messages;
import ej.websocket.ReasonForClosure;

/**
 * This build will create frames that a client can send. That is, it will mask the payload, set the MASK bit and add the masking to the frame.
 */
public class ClientFrameBuilder implements IFrameBuilder {

	// We need to generate random bytes to create masks
	private static final Random RANDOM = new Random();

	/**
	 * @param opcode
	 *            the desired opcode. It will be masked to add the FIN bit.
	 * @param payload
	 *            the payload of the frame. MUST BE NOT NULL.
	 * @return a byte array that contains the whole frame
	 */
	private static byte[] buildGenericFrame(int opcode, final byte[] payload) {
		int frameLength = computeFrameLength(payload);
		byte[] bytes = new byte[frameLength];
		int offset = 0;

		// Set opcode
		bytes[offset] = (byte) ((byte) (opcode & 0xFF) | 0x80);
		offset += 1;

		// Set the byte with MASK (1 bit) and payload length (7 bits), optionally add extended payload length
		int payloadLength = payload.length;

		if (payloadLength <= 125) {
			// Case when payload length is 7 bit wide
			bytes[offset] = (byte) (0x80 | payloadLength);
			offset += 1; // one mandatory and no optional byte are added

		} else if (payloadLength <= EXTENDED_LENGTH_MAX_VALUE) {
			// Case when payload length is 7+16 bit wide
			bytes[offset] = (byte) (0x80 | 126);
			bytes[offset + 1] = (byte) ((payloadLength & 0xFF00) >> 8); // MSB first
			bytes[offset + 2] = (byte) ((payloadLength & 0x00FF) >> 0);
			offset += 3; // one mandatory and two optional bytes are added

		} else {
			// Case when payload length is 7+64 bit wide
			bytes[offset] = (byte) (0x80 | 127);
			// bytes[2..5] = ?
			offset += 5; // one mandatory and four optional bytes are added
			// TODO future versions may handle very long frames
			// "Very extended frames are not handled by this implementation".
			throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
					Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
		}

		// Generate mask
		byte[] mask = new byte[4];
		RANDOM.nextBytes(mask);
		// FIXME we will probably need a stronger source of entropy to generate random bytes

		// Add mask to frame bytes
		bytes[offset + 0] = mask[0];
		bytes[offset + 1] = mask[1];
		bytes[offset + 2] = mask[2];
		bytes[offset + 3] = mask[3];
		offset += 4; // four mandatory bytes are added

		// Add masked payload
		for (int i = 0; i < payloadLength; i++) {
			// If payload is null, then length equals zero and we will never enter this loop
			bytes[offset + i] = (byte) (payload[i] ^ mask[i % 4]);
		}

		return bytes;
	}

	/**
	 * Equivalent to calling <code>computeFrameLength(payload.length)</code>.
	 *
	 * @param payload
	 *            payload to send
	 * @return the total length (in bytes) of the corresponding frame
	 */
	/* default */static int computeFrameLength(byte[] payload) {
		return computeFrameLength(payload.length);
	}

	/**
	 * @param payloadLength
	 *            actual length of the payload to send
	 * @return the total length (in bytes) of the corresponding frame
	 */
	/* default */static int computeFrameLength(int payloadLength) {
		// Fixed part: 2 bytes for header, 4 bytes for mask
		int frameLength = 2 + 4;

		// Variable part: payload length, optional extended payload length, payload itself
		if (payloadLength <= 125) {
			frameLength += payloadLength;

		} else if (payloadLength <= (2 << 16 - 1)) {
			frameLength += 2; // 2 bytes for extended payload length
			frameLength += payloadLength;

		} else {
			// TODO future versions may handle very long frames
			// "Very extended frames are not handled by this implementation".
			throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
					Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
		}

		// That's it
		return frameLength;
	}

	@Override
	public RawFrame buildBinaryFrame(byte[] binary) {
		// Create payload
		byte[] payload = binary == null ? new byte[0] : binary;

		// Create frame
		byte[] bytes = buildGenericFrame(RawFrame.OPCODE_BINARY_FRAME, payload);
		return new RawFrame(bytes);
	}

	@Override
	public RawFrame buildCloseFrame(ReasonForClosure reasonForClosure) {
		// Create payload
		byte[] payload = new byte[0];

		// Create payload
		int length = 2 + reasonForClosure.getCloseReason().length();
		payload = new byte[length];

		int code = reasonForClosure.getCloseCode();
		payload[0] = (byte) ((code & 0xFF00) >> 8);
		payload[1] = (byte) ((code & 0x00FF) >> 0);

		System.arraycopy(reasonForClosure.getCloseReason().getBytes(), 0, payload, 2, length - 2);

		// Create frame
		byte[] bytes = buildGenericFrame(RawFrame.OPCODE_CONNECTION_CLOSE, payload);
		return new RawFrame(bytes);
	}

	@Override
	public RawFrame buildPingFrame(byte[] binary) {

		// Create payload
		byte[] payload = binary == null ? new byte[0] : binary;

		// Create frame
		byte[] bytes = buildGenericFrame(RawFrame.OPCODE_PING, payload);
		return new RawFrame(bytes);
	}

	@Override
	public RawFrame buildPongFrame(byte[] binary) {
		// Create payload
		byte[] payload = binary == null ? new byte[0] : binary;

		// Create frame
		byte[] bytes = buildGenericFrame(RawFrame.OPCODE_PONG, payload);
		return new RawFrame(bytes);
	}

	@Override
	public RawFrame buildTextFrame(String text) {
		// Create payload
		byte[] payload = text == null ? new byte[0] : text.getBytes();

		// Create frame
		byte[] bytes = buildGenericFrame(RawFrame.OPCODE_TEXT_FRAME, payload);
		return new RawFrame(bytes);
	}

}
