/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.paint;

import java.awt.Color;
import java.awt.Paint;

/**
 * Painter to apply when drawing a path.
 */
public interface VGPaint {

	/**
	 * Value that indicates a fully opaque opacity.
	 */
	int ALPHA_FULLY_OPAQUE = 0xff;
	/**
	 * Value that indicates a fully transparent opacity.
	 */
	int ALPHA_FULLY_TRANSPARENT = 0x00;
	/**
	 * Value that to shift the opacity value in a ARGB 32-bit color.
	 */
	int ALPHA_SHIFT = 24;

	/**
	 * Creates an AWT {@link Paint} object.
	 *
	 * @return a AWT Paint
	 */
	Paint getPaint();

	/**
	 * Creates a new painter applying the given visitor. The original painter is not modified.
	 *
	 * @param visitor
	 *            the visitor to apply
	 * @return a new instance of {@link VGPaint}
	 */
	VGPaint apply(PaintVisitor visitor);

	/**
	 * Utility method that returns a new color applying an opacity on the given color.
	 *
	 * @param color
	 *            a ARGB color
	 * @param alpha
	 *            the opacity to apply
	 * @return a new color
	 */
	static int applyOpacity(int color, int alpha) {
		int ac = (((color >> ALPHA_SHIFT) & 0xff) * alpha) / 255;
		return (color & ~(ALPHA_FULLY_OPAQUE << ALPHA_SHIFT)) | (ac << ALPHA_SHIFT);
	}

	/**
	 * Utility method that returns a new color applying an opacity on the given color.
	 *
	 * @param color
	 *            the source color
	 * @param alpha
	 *            the opacity to apply
	 * @return a new color
	 */
	static int applyOpacity(Color color, int alpha) {
		return applyOpacity(color.getRGB(), alpha);
	}
}
