/*
 * Copyright 2019-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.storage.heap;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Set;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.storage.Storage;
import ej.storage.util.StorageHelper;

/**
 * Implementation of not persistent storage on Java heap.
 */
public class StorageHeap implements Storage {

	private final PackedMap<String, byte[]> data;

	/**
	 * Creates a not persistent storage.
	 */
	public StorageHeap() {
		this.data = new PackedMap<>();
	}

	@Override
	public OutputStream store(String id) throws IOException {
		StorageHelper.checkID(id);
		return createOutputStream(id, true);
	}

	private OutputStream createOutputStream(final String id, final boolean overwriteData) {
		return new ByteArrayOutputStream() {
			private boolean overwriteFirstFlush = overwriteData;
			private boolean closed;

			@Override
			public void flush() throws IOException {
				if (!this.closed) {
					boolean overwriteFirstFlush;
					synchronized (this) {
						overwriteFirstFlush = this.overwriteFirstFlush;
						this.overwriteFirstFlush = false;
					}
					if (overwriteFirstFlush) {
						StorageHeap.this.data.remove(id);
					}

					byte[] oldData = StorageHeap.this.data.get(id);
					byte[] newData = toByteArray();
					byte[] data;
					if (oldData != null) {
						int oldDataLength = oldData.length;
						int newDataLength = newData.length;
						data = new byte[oldDataLength + newDataLength];
						System.arraycopy(oldData, 0, data, 0, oldDataLength);
						System.arraycopy(newData, 0, data, oldDataLength, newDataLength);
					} else {
						data = newData;
					}
					StorageHeap.this.data.put(id, data);
				}
				reset();
			}

			@Override
			public void close() throws IOException {
				flush();
				this.closed = true;
			}
		};
	}

	@Override
	public OutputStream modify(String id, int offset) throws IOException {
		StorageHelper.checkID(id);
		StorageHelper.checkOffset(offset);
		return createOutputStreamModify(id, offset);
	}

	private OutputStream createOutputStreamModify(final String id, final int offset) {
		return new ByteArrayOutputStream() {
			private int position = offset;
			private boolean closed;

			@Override
			public void flush() throws IOException {
				if (!this.closed) {
					byte[] oldData = StorageHeap.this.data.get(id);
					byte[] newData = toByteArray();
					int position = this.position;
					int oldDataLength = (oldData == null ? 0 : oldData.length);
					int newDataLength = newData.length;
					int dataLength = Math.max(oldDataLength, position + newDataLength);
					byte[] data = new byte[dataLength];
					if (oldData != null) {
						System.arraycopy(oldData, 0, data, 0, oldDataLength);
					}
					System.arraycopy(newData, 0, data, position, newDataLength);
					StorageHeap.this.data.put(id, data);
					this.position += newDataLength;
				}
				reset();
			}

			@Override
			public void close() throws IOException {
				flush();
				this.closed = true;
			}
		};
	}

	@Override
	public OutputStream append(String id) throws IOException {
		StorageHelper.checkID(id);
		return createOutputStream(id, false);
	}

	@Override
	@Nullable
	public InputStream load(String id) throws IOException {
		StorageHelper.checkID(id);
		byte[] bytes = this.data.get(id);
		if (bytes == null) {
			return null;
		} else {
			return new ByteArrayInputStream(bytes);
		}
	}

	@Override
	public void move(String sourceId, String destinationId) throws IOException {
		StorageHelper.checkID(sourceId);
		StorageHelper.checkID(destinationId);
		byte[] data = this.data.remove(sourceId);
		if (data != null) {
			this.data.put(destinationId, data);
		} else {
			throw new IOException();
		}
	}

	@Override
	public void remove(String id) throws IOException {
		StorageHelper.checkID(id);
		byte[] data = this.data.remove(id);
		if (data == null) {
			throw new IOException();
		}
	}

	@Override
	public long getSize(String id) throws IOException {
		StorageHelper.checkID(id);
		byte[] data = this.data.get(id);
		if (data != null) {
			return data.length;
		} else {
			throw new IOException();
		}
	}

	@Override
	public boolean exists(String id) throws IOException {
		StorageHelper.checkID(id);
		return this.data.get(id) != null;
	}

	@Override
	public String[] getIds() throws IOException {
		Set<String> keySet = this.data.keySet();
		return keySet.toArray(new String[keySet.size()]);
	}

}
