/*
 * Java
 *
 * 2010-2020 ESR - Not subject to Copyright.
 * This document has been released and published by E-S-R consortium, a non-profit entity.
 * To learn more about E-S-R consortium, please visit http://www.e-s-r.net/.
 * The matter contained in this document is not subject to copyright; you are free to use it for any purpose, for more information see E-S-R consortium policies.
 */
package ej.microui.event;

import ej.annotation.Nullable;
import ej.microui.MicroUIException;

/**
 * Event generators generate int-based events (see {@link Event}). They are responsible for holding data that cannot be
 * sent within the event. Event generators are state machines that convert serialized (mostly external) integers to
 * MicroUI events.<br>
 * <p>
 * There is a system pool that holds generators. A generator in the system pool has an identifier between 0 and 254,
 * otherwise identifier is 0xFF. The advantage of putting a generator in the system pool is that it can then be
 * looked-up using {@link #get(int)} and {@link #get(Class, int)}.
 *
 * @see Event
 */

public abstract class EventGenerator {

	/**
	 * Creates a new event generator. As soon as a new event generator is added to the system pool it has a valid
	 * identifier.
	 */
	public EventGenerator() {
		throw new RuntimeException();
	}

	/**
	 * Adds the generator in the system pool. The generator can only be added once as a system generator. When added,
	 * its identifier is set.
	 *
	 * @return the event generator's identifier in the system pool.
	 * @throws RuntimeException
	 *             if the maximum number of event generators added to the system pool has been reached
	 */
	public int addToSystemPool() {
		throw new RuntimeException();
	}

	/**
	 * Removes the generator from the system generators pool. If the generator is not in the pool, nothing is done.
	 */
	public void removeFromSystemPool() {
		throw new RuntimeException();
	}

	/**
	 * Sets an event handler to this event generator. It will receive the generated events.
	 * <p>
	 * It replaces the old one and can be <code>null</code>.
	 *
	 * @param eventHandler
	 *            the event handler to add.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to handle events from this event
	 *             generator.
	 * @since 2.0
	 */
	public void setEventHandler(@Nullable EventHandler eventHandler) {
		throw new RuntimeException();
	}

	/**
	 * Gets a generator from its identifier.
	 *
	 * @param id
	 *            the generator's identifier.
	 * @return the associated event generator.
	 * @throws IndexOutOfBoundsException
	 *             if the generator does not exist.
	 * @throws MicroUIException
	 *             if MicroUI is not started
	 */
	public static EventGenerator get(int id) {
		throw new RuntimeException();
	}

	/**
	 * Gets a generator whose class is <code>clazz</code> from the system pool starting the search from
	 * <code>fromIndex</code>. If class <code>clazz</code> is not an <code>EventGenerator</code> or if nothing is found,
	 * returns <code>null</code>.
	 *
	 * @param clazz
	 *            the <code>EventGenerator</code>'s class to return
	 * @param fromIndex
	 *            index from which starting to search
	 * @return the <code>EventGenerator</code> or <code>null</code>.
	 * @throws MicroUIException
	 *             if MicroUI is not started
	 */
	public static @Nullable <E extends EventGenerator> E get(Class<E> clazz, int fromIndex) {
		throw new RuntimeException();
	}

	/**
	 * Gets all generators of the given type from the system pool in an array.
	 *
	 * @param clazz
	 *            the type of the event generators to return.
	 * @return the arrays of event generators.
	 * @throws MicroUIException
	 *             if MicroUI is not started
	 */
	public static <E extends EventGenerator> EventGenerator[] get(Class<E> clazz) {
		throw new RuntimeException();
	}

	/**
	 * Gets the generator's unique identifier. Up to 254 <code>EventGenerator</code>s can be installed as MicroUI system
	 * event generator.
	 *
	 * @return the generator's identifier as an <code>int</code>.
	 */
	public final int getId() {
		throw new RuntimeException();
	}

	/**
	 * Gets the generator's event handler.
	 *
	 * @return the generator's event handler.
	 * @since 2.0
	 */
	public @Nullable EventHandler getEventHandler() {
		throw new RuntimeException();
	}

	/**
	 * Gets the event type associated with the event generator
	 *
	 * @return the event type
	 */
	public abstract int getEventType();

	/**
	 * Sends the given event to the current {@link EventHandler}. Does nothing if there is no event handler set to this
	 * event generator.
	 *
	 * @param event
	 *            the event to send
	 */
	protected void sendEvent(int event) {
		throw new RuntimeException();
	}
}
