/*******************************************************************************
 * Copyright (c) 2009, 2018 IBM Corp.
 * Copyright 2019-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * and Eclipse Distribution License v1.0 which accompany this distribution.
 *
 * The Eclipse Public License is available at
 *    http://www.eclipse.org/legal/epl-v10.html
 * and the Eclipse Distribution License is available at
 *   http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * Contributors:
 *    Dave Locke - initial API and implementation and/or initial documentation
 *    Ian Craggs - MQTT 3.1.1 support
 *    Ian Craggs - per subscription message handlers (bug 466579)
 *    Ian Craggs - ack control (bug 472172)
 *    MicroEJ Corp. - MicroPaho implementation and optimizations on MicroEJ
 */

package org.eclipse.paho.client.mqttv3;

/**
 * Enables an application to communicate with an MQTT server using blocking methods.
 * <p>
 * This interface allows applications to utilize all features of the MQTT version 3.1 specification including:
 * </p>
 * <ul>
 * <li>connect
 * <li>publish
 * <li>subscribe
 * <li>unsubscribe
 * <li>disconnect
 * </ul>
 * <p>
 * IMqttClient provides a set of methods that block and return control to the application program once the MQTT action
 * has completed.
 * </p>
 */
public interface IMqttClient extends AutoCloseable {
	/**
	 * Connects to an MQTT server using the default options.
	 * <p>
	 * The default options are specified in {@link MqttConnectOptions} class.
	 * </p>
	 *
	 * @throws MqttException
	 *             for non security related problems
	 * @see #connect(MqttConnectOptions)
	 */
	public void connect() throws MqttException;

	/**
	 * Connects to an MQTT server using the specified options.
	 * <p>
	 * The server to connect to is specified on the constructor. It is recommended to call
	 * {@link #setCallback(MqttCallback)} prior to connecting in order that messages destined for the client can be
	 * accepted as soon as the client is connected.
	 * </p>
	 * <p>
	 * This is a blocking method that returns once connect completes
	 * </p>
	 *
	 * @param options
	 *            a set of connection parameters that override the defaults.
	 * @throws MqttException
	 *             for non security related problems including communication errors
	 */
	public void connect(MqttConnectOptions options) throws MqttException;

	/**
	 * Disconnects from the server.
	 *
	 * <p>
	 * This is a blocking method that returns once disconnect completes
	 * </p>
	 *
	 * @throws MqttException
	 *             if a problem is encountered while disconnecting
	 */
	public void disconnect() throws MqttException;

	/**
	 * Subscribe to a topic, which may include wildcards using a QoS of 1.
	 *
	 * @param topicFilter
	 *            the topic to subscribe to, which can include wildcards.
	 * @throws MqttException
	 *             if there was an error registering the subscription.
	 */
	public void subscribe(String topicFilter) throws MqttException;

	/**
	 * Subscribe to a topic, which may include wildcards.
	 *
	 * @param topicFilter
	 *            the topic to subscribe to, which can include wildcards.
	 * @param qos
	 *            the maximum quality of service at which to subscribe. Messages published at a lower quality of service
	 *            will be received at the published QoS. Messages published at a higher quality of service will be
	 *            received using the QoS specified on the subscribe.
	 * @throws MqttException
	 *             if there was an error registering the subscription.
	 */
	public void subscribe(String topicFilter, int qos) throws MqttException;

	/**
	 * Requests the server unsubscribe the client from a topic.
	 *
	 * @param topicFilter
	 *            the topic to unsubscribe from. It must match a topicFilter specified on the subscribe.
	 * @throws MqttException
	 *             if there was an error unregistering the subscription.
	 */
	public void unsubscribe(String topicFilter) throws MqttException;

	/**
	 * Publishes a message to a topic on the server and return once it is delivered.
	 * <p>
	 * This is a convenience method, which will create a new {@link MqttMessage} object with a byte array payload and
	 * the specified QoS, and then publish it. All other values in the message will be set to the defaults.
	 * </p>
	 *
	 * @param topic
	 *            to deliver the message to, for example "finance/stock/ibm".
	 * @param payload
	 *            the byte array to use as the payload
	 * @param qos
	 *            the Quality of Service to deliver the message at. Valid values are 0, 1 or 2.
	 * @param retained
	 *            whether or not this message should be retained by the server.
	 * @throws IllegalArgumentException
	 *             if value of QoS is not 0, 1 or 2.
	 * @throws MqttException
	 *             for other errors encountered while publishing the message. For instance client not connected.
	 * @see #publish(String, MqttMessage)
	 * @see MqttMessage#setQos(int)
	 * @see MqttMessage#setRetained(boolean)
	 */
	public void publish(String topic, byte[] payload, int qos, boolean retained) throws MqttException;

	/**
	 * Publishes a message to a topic on the server.
	 * <p>
	 * Delivers a message to the server at the requested quality of service and returns control once the message has
	 * been delivered.
	 * </p>
	 *
	 * <p>
	 * This is a blocking method that returns once publish completes
	 * </p>
	 *
	 * @param topic
	 *            to deliver the message to, for example "finance/stock/ibm".
	 * @param message
	 *            to delivery to the server
	 * @throws MqttException
	 *             for other errors encountered while publishing the message. For instance client not connected.
	 */
	public void publish(String topic, MqttMessage message) throws MqttException;

	/**
	 * Sets the callback listener to use for events that happen asynchronously.
	 * <p>
	 * There are a number of events that listener will be notified about. These include:
	 * </p>
	 * <ul>
	 * <li>A new message has arrived and is ready to be processed</li>
	 * <li>The connection to the server has been lost</li>
	 * <li>Delivery of a message to the server has completed.</li>
	 * </ul>
	 *
	 * @see MqttCallback
	 * @param callback
	 *            the class to callback when for events related to the client
	 */
	public void setCallback(MqttCallback callback);

	/**
	 * Determines if this client is currently connected to the server.
	 *
	 * @return <code>true</code> if connected, <code>false</code> otherwise.
	 */
	public boolean isConnected();

	/**
	 * Returns the client ID used by this client.
	 * <p>
	 * All clients connected to the same server or server farm must have a unique ID.
	 * </p>
	 *
	 * @return the client ID used by this client.
	 */
	public String getClientId();

	/**
	 * Returns the address of the server used by this client, as a URI.
	 * <p>
	 * The format is the same as specified on the constructor.
	 * </p>
	 *
	 * @return the server's address, as a URI String.
	 * @see MqttClient#MqttClient(String, String)
	 */
	public String getServerURI();

	/**
	 * Close the client Releases all resource associated with the client. After the client has been closed it cannot be
	 * reused. For instance attempts to connect will fail.
	 *
	 * @throws MqttException
	 *             if the client is not disconnected.
	 */
	@Override
	public void close() throws MqttException;
}
