/*
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.kf;

import java.io.InputStream;

/**
 * <p>
 * A {@link Converter} is able to give a representation of an object owned by a Feature to another Feature. A
 * {@link Converter} is able to convert instances of one and only one Kernel type.
 * </p>
 * <p>
 * A {@link Converter} is free to decide the kind of conversion to apply, depending on the managed type. For example, a
 * {@link Converter} for immutables instances of types such as {@link String} will most likely return a copy (clone),
 * whereas a {@link Converter} for instances of types such as {@link InputStream} will most likely return a wrapper on
 * the original object.
 * </p>
 *
 * @param <T>
 *            the Kernel type managed by this {@link Converter}.
 * @see Kernel#bind(Object, Class, Feature)
 * @see Kernel#clone(Object, Module)
 */
public interface Converter<T> {

	/**
	 * Converts an object owned by a Feature to another Feature.
	 *
	 * @param source
	 *            the source object to be converted
	 * @param targetOwner
	 *            the owner of the converted object
	 * @return the converted object, owned by the target owner
	 */
	public T convert(T source, Feature targetOwner);

	/**
	 * Gets the Kernel type managed by this {@link Converter}.
	 *
	 * @return the Kernel type managed by this {@link Converter}.
	 */
	public Class<T> getType();
}
