/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.fp.FrontPanel;

/**
 * This class holds all methods to send MicroUI Command events to MicroUI application.
 */
@SuppressWarnings("nls")
public class EventCommand {

	/**
	 * Usual MicroUI Command generator tag used in platform configuration project (<code>microui.xml</code>).
	 */
	public static final String COMMON_MICROUI_GENERATOR_TAG = "COMMANDS";

	/**
	 * The "escape" command constant.
	 * <p>
	 * The value <code>0x0000</code> is assigned to <code>ESC</code>.
	 */
	public static final int ESC = 0x0000;
	/**
	 * The "back" command constant.
	 * <p>
	 * The value <code>0x0001</code> is assigned to <code>BACK</code>.
	 */
	public static final int BACK = 0x0001;
	/**
	 * The "up" command constant.
	 * <p>
	 * The value <code>0x0002</code> is assigned to <code>UP</code>.
	 */
	public static final int UP = 0x0002;
	/**
	 * The "down" command constant.
	 * <p>
	 * The value <code>0x0003</code> is assigned to <code>DOWN</code>.
	 */
	public static final int DOWN = 0x0003;
	/**
	 * The "left" command constant.
	 * <p>
	 * The value <code>0x0004</code> is assigned to <code>LEFT</code>.
	 */
	public static final int LEFT = 0x0004;
	/**
	 * The "right" command constant.
	 * <p>
	 * The value <code>0x0005</code> is assigned to <code>RIGHT</code>.
	 */
	public static final int RIGHT = 0x0005;
	/**
	 * The "select" command constant.
	 * <p>
	 * The value <code>0x0006</code> is assigned to <code>SELECT</code>.
	 */
	public static final int SELECT = 0x0006;
	/**
	 * The "cancel" command constant.
	 * <p>
	 * The value <code>0x0007</code> is assigned to <code>CANCEL</code>.
	 */
	public static final int CANCEL = 0x0007;
	/**
	 * The "help" command constant.
	 * <p>
	 * The value <code>0x0008</code> is assigned to <code>HELP</code>.
	 */
	public static final int HELP = 0x0008;

	/**
	 * The "menu" command constant.
	 * <p>
	 * The value <code>0x0009</code> is assigned to <code>MENU</code>.
	 */
	public static final int MENU = 0x0009;

	/**
	 * The "exit" command constant.
	 * <p>
	 * The value <code>0x000A</code> is assigned to <code>EXIT</code>.
	 */
	public static final int EXIT = 0x000A;

	/**
	 * The "start" command constant.
	 * <p>
	 * The value <code>0x000B</code> is assigned to <code>START</code>.
	 */
	public static final int START = 0x000B;

	/**
	 * The "stop" command constant.
	 * <p>
	 * The value <code>0x000C</code> is assigned to <code>STOP</code>.
	 */
	public static final int STOP = 0x000C;

	/**
	 * The "pause" command constant.
	 * <p>
	 * The value <code>0x000D</code> is assigned to <code>PAUSE</code>.
	 */
	public static final int PAUSE = 0x000D;

	/**
	 * The "resume" command constant.
	 * <p>
	 * The value <code>0x000E</code> is assigned to <code>RESUME</code>.
	 */
	public static final int RESUME = 0x000E;

	/**
	 * The "copy" command constant.
	 * <p>
	 * The value <code>0x000F</code> is assigned to <code>COPY</code>.
	 */
	public static final int COPY = 0x000F;

	/**
	 * The "cut" command constant.
	 * <p>
	 * The value <code>0x0010</code> is assigned to <code>CUT</code>.
	 */
	public static final int CUT = 0x0010;

	/**
	 * The "paste" command constant.
	 * <p>
	 * The value <code>0x0011</code> is assigned to <code>PASTE</code>.
	 */
	public static final int PASTE = 0x0011;

	/**
	 * The "clockwise" command constant.
	 * <p>
	 * The value <code>0x0012</code> is assigned to <code>CLOCKWISE</code>.
	 */
	public static final int CLOCKWISE = 0x0012;

	/**
	 * The "anti-clockwise" command constant.
	 * <p>
	 * The value <code>0x0013</code> is assigned to <code>ANTICLOCKWISE</code>.
	 */
	public static final int ANTICLOCKWISE = 0x0013;

	/**
	 * The "previous" command constant.
	 * <p>
	 * The value <code>0x0014</code> is assigned to <code>PREVIOUS</code>.
	 */
	public static final int PREVIOUS = 0x0014;

	/**
	 * The "next" command constant.
	 * <p>
	 * The value <code>0x0015</code> is assigned to <code>NEXT</code>.
	 */
	public static final int NEXT = 0x0015;

	/**
	 * The "display" command constant.
	 * <p>
	 * The value <code>0x0016</code> is assigned to <code>DISPLAY</code>.
	 */
	public static final int DISPLAY = 0x0016;

	// MicroUI command representation
	private static final int COMMAND_DRIVER_COMMAND_MASK = 0x0000ffff;
	private static final int COMMAND_DRIVER_COMMAND_SHIFT = 0;

	private EventCommand() {
		// hide implicit constructor
	}

	/**
	 * Send a command event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param command
	 *            the command to send, between 0 and 255.
	 *
	 * @see #sendEvent(String, int)
	 */
	public static void sendEvent(int command) {
		sendEvent(COMMON_MICROUI_GENERATOR_TAG, command);
	}

	/**
	 * Send a command event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Command</code> event
	 * generator.
	 *
	 * @param commandsEventGenerator
	 *            the MicroUI <code>Command</code> event generator.
	 * @param command
	 *            the command to send, between 0 and 255.
	 *
	 */
	public static void sendEvent(String commandsEventGenerator, int command) {
		try {
			LLUIInput.sendEvent(commandsEventGenerator,
					((command << COMMAND_DRIVER_COMMAND_SHIFT) & COMMAND_DRIVER_COMMAND_MASK));

		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(commandsEventGenerator, command);
		}
	}

	private static void traceEvent(String commandsEventGenerator, int command) {
		StringBuilder buf = new StringBuilder();
		buf.append("[");
		buf.append(EventCommand.class.getName());
		buf.append("] ");
		buf.append("command 0x");
		buf.append(Integer.toHexString(command));
		buf.append(" (");
		buf.append(commandsEventGenerator);
		buf.append(")");
		FrontPanel.out.println(buf.toString());
	}
}
