/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth.listeners;

import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;

/**
 * The <code>RemoteServiceListener</code> interface provides callbacks related to remote services.
 * <p>
 * After {@link ej.bluetooth.BluetoothConnection#discoverServices discovering the services} of a device, the application may
 * {@link ej.bluetooth.BluetoothService#setRemoteListener set the remote listener} of a service in order to be notified
 * of the events related to this service.
 * <p>
 * This library provides a default implementation of this interface:
 * {@link ej.bluetooth.listeners.impl.DefaultRemoteServiceListener}.
 */
public interface RemoteServiceListener {

	/**
	 * Called when a response to a read request is received.
	 *
	 * @param connection
	 *            the connection with the device which has sent the response.
	 * @param attribute
	 *            the attribute which was read.
	 * @param status
	 *            the response status (see {@link ej.bluetooth.BluetoothStatus}).
	 * @param value
	 *            the value of the attribute.
	 * @see ej.bluetooth.BluetoothConnection#sendReadRequest(BluetoothAttribute)
	 */
	void onReadCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status, byte[] value);

	/**
	 * Called when a response to a write request is received.
	 *
	 * @param connection
	 *            the connection with the device which has sent the response.
	 * @param attribute
	 *            the attribute which was written.
	 * @param status
	 *            the response status (see {@link ej.bluetooth.BluetoothStatus}).
	 * @see ej.bluetooth.BluetoothConnection#sendWriteRequest(BluetoothAttribute, byte[])
	 */
	void onWriteCompleted(BluetoothConnection connection, BluetoothAttribute attribute, byte status);

	/**
	 * Called when a characteristic notification or indication is received.
	 *
	 * @param connection
	 *            the connection with the device which has sent the notification.
	 * @param characteristic
	 *            the characteristic which has changed.
	 * @param value
	 *            the value of the characteristic.
	 */
	void onNotificationReceived(BluetoothConnection connection, BluetoothCharacteristic characteristic, byte[] value);
}
