/*
 * Java
 *
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.rcommand.serversocket;

import java.io.IOException;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.NetworkInterface;
import java.net.ServerSocket;
import java.net.Socket;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.logging.Logger;

import ej.rcommand.RemoteCommandManager;
import ej.rcommand.RemoteConnection;
import ej.rcommand.connectivity.ConnectivityListener;
import ej.rcommand.connectivity.ConnectivityManager;

/**
 * Server which client administration connect to.
 */
public class RemoteCommandServer implements Runnable, ConnectivityManager {
	private final Logger logger;
	private final int port;
	private int nbConnections;
	private final int nbMaxConnections;
	private final Object monitor;
	private volatile boolean running;
	private final RemoteCommandManager commandManager;
	private ServerSocket serverSocket;
	private final List<ConnectivityListener> listeners;

	/**
	 * Creates a server which listen to the given port and accepts at maximum
	 * the given number of connections.
	 *
	 * @param commandManager
	 *            the command manager.
	 * @param port
	 *            the port where the server listen.
	 * @param nbMaxConnection
	 *            the maximum number of simultaneous connection.
	 */
	public RemoteCommandServer(RemoteCommandManager commandManager, int port, int nbMaxConnection) {
		this.logger = Logger.getLogger(RemoteCommandServer.class.getSimpleName());
		this.port = port;
		this.nbMaxConnections = nbMaxConnection;
		this.nbConnections = 0;
		this.commandManager = commandManager;
		this.monitor = new Object();
		this.listeners = new ArrayList<>();
	}

	@Override
	public void run() {
		Socket client;
		this.running = true;
		int delay = 250;

		do {
			boolean networkAvailable = false;
			try (ServerSocket serverSocket = new ServerSocket()) {
				serverSocket.setReuseAddress(true);
				serverSocket.bind(new InetSocketAddress((InetAddress) null, this.port));
				networkAvailable = true;
				// reset the delay to zero
				delay = 0;
				this.serverSocket = serverSocket;
				this.logger.info("Server listening on port " + this.port); //$NON-NLS-1$
				while (this.running) {
					synchronized (this.monitor) {
						while (this.nbConnections == this.nbMaxConnections) {
							this.monitor.wait();
						}
					}
					client = serverSocket.accept();
					RemoteConnection remoteConnection = new SocketRemoteConnection(client, this);
					this.commandManager.startReading(remoteConnection, "Admin " + client.getRemoteSocketAddress());
					this.nbConnections++;
					notifyConnectivityListener(true);
				}
			} catch (IOException e) {
				// do nothing
			} catch (InterruptedException e) {
				// do nothing
			}
			if (this.running && !networkAvailable) {
				try {
					this.logger.fine("Waiting for " + delay + " ms for the network."); //$NON-NLS-1$
					Thread.sleep(delay);
				} catch (InterruptedException e) {
					e.printStackTrace();
					Thread.interrupted();
					return;
				}
				// next time, double the delay, but keep it under 10 seconds
				if (delay < 10000) {
					delay *= 2;
				}
			}
		} while (this.running);
	}

	public void stopRunning() throws IOException {
		if (this.serverSocket != null) {
			this.serverSocket.close();
		}

		synchronized (this.monitor) {
			this.monitor.notifyAll();
		}

		while (this.nbConnections > 0) {
			this.nbConnections--;
			notifyConnectivityListener(false);
		}
	}

	/* package */void releaseConnection() {
		synchronized (this.monitor) {
			this.nbConnections--;
			this.monitor.notify();
		}
		notifyConnectivityListener(false);
	}

	@Override
	public int getAvailable() {
		synchronized (this.monitor) {
			return this.nbConnections;
		}
	}

	@Override
	public String getConnectionInfo() {
		if (this.serverSocket != null) {
			try {
				Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
				if (interfaces != null) {
					StringBuffer info = new StringBuffer();
					while (interfaces.hasMoreElements()) {
						NetworkInterface iface = interfaces.nextElement();
						// filters out 127.0.0.1 and inactive interfaces
						if (iface.isLoopback() || !iface.isUp()) {
							continue;
						}

						Enumeration<InetAddress> addresses = iface.getInetAddresses();
						int n = 0;
						while (addresses.hasMoreElements()) {
							if (n++ == 1) {
								info.append(',');
							}
							info.append(addresses.nextElement().getHostAddress());
						}
						info.append(':').append(this.port);
						return info.toString();
					}
				}
			} catch (SocketException e) {
				// cannot retrieve the information
			}
		}
		return null;
	}

	private synchronized void notifyConnectivityListener(boolean connect) {
		if (connect) {
			for (ConnectivityListener listener : this.listeners) {
				listener.onConnect();
			}
		} else {
			for (ConnectivityListener listener : this.listeners) {
				listener.onDisconnect();
			}
		}
	}

	@Override
	public synchronized void addConnectivityListener(ConnectivityListener listener) {
		if (!this.listeners.contains(listener)) {
			this.listeners.add(listener);
		}
	}

	@Override
	public synchronized void removeConnectivityListener(ConnectivityListener listener) {
		this.listeners.remove(listener);
	}
}
