/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Container;
import ej.mwt.Widget;

/**
 * An even child selector selects by checking if a widget is at an even position in its parent (1, 3, 5,…).
 * <p>
 * Equivalent to <code>:nth-child(even)</code> selector in CSS. Its specificity is (0,0,1,0).
 *
 * @see Widget#getParent()
 * @see SelectorHelper
 */
public class EvenChildSelector implements Selector {

	/**
	 * Even child selector singleton to avoid creating several ones.
	 */
	public static final EvenChildSelector EVEN_CHILD_SELECTOR = new EvenChildSelector();

	/**
	 * Creates an even child selector.
	 */
	private EvenChildSelector() {
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		Container parent = widget.getParent();
		if (parent == null) {
			return false;
		}

		int index = parent.getChildIndex(widget);
		// /!\ Beware, the first index of an array is 0 (odd became even).
		return (index & 1) == 1;
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 1, 0);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		return (obj == this);
	}

	@Override
	public int hashCode() {
		return 0;
	}
}
