/*
 * Java
 *
 * Copyright 2019-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth.commands;

import java.io.IOException;

import ej.annotation.Nullable;
import ej.bluetooth.BluetoothConnection;
import ej.bluetooth.BluetoothUuid;
import ej.library.iot.rcommand.bluetooth.BluetoothController;
import ej.library.iot.rcommand.bluetooth.Commands;
import ej.rcommand.CommandReader;
import ej.rcommand.CommandSender;
import ej.rcommand.synchronous.Endpoint;

public class DiscoverServicesEndpoint implements Endpoint {

	private final BluetoothController controller;

	private short connHandle;
	private @Nullable byte[] uuid;

	public DiscoverServicesEndpoint(BluetoothController controller) {
		this.controller = controller;
	}

	@Override
	public String getName() {
		return Commands.BLUETOOTH_DISCOVER_SERVICES;
	}

	@Override
	public void readRequestBody(CommandReader reader) throws IOException {
		this.connHandle = (short) reader.readInt();

		byte[] uuid = reader.readByteArray();
		this.uuid = (uuid.length != 0 ? uuid : null);
	}

	@Override
	public void writeResponseBody(CommandSender sender) throws IOException {
		BluetoothConnection connection = this.controller.getConnection(this.connHandle);
		boolean success;
		if (connection == null) {
			success = false;
		} else {
			byte[] uuid = this.uuid;
			if (uuid == null) {
				success = connection.discoverServices();
			} else {
				BluetoothUuid uuidObject = new BluetoothUuid(uuid, 0);
				success = connection.discoverService(uuidObject);
			}
		}

		if (success) {
			sender.sendInt(Commands.OK);
		} else {
			sender.sendInt(Commands.ERROR);
		}
	}
}
