/*
 * Decompiled with CFR 0.152.
 */
package java.time;

import ej.annotation.Nullable;
import java.io.Serializable;
import java.time.BigInteger;
import java.time.DateTimeException;
import java.time.MathUtils;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.time.temporal.TemporalAmount;
import java.time.temporal.TemporalUnit;
import java.time.temporal.UnsupportedTemporalTypeException;
import java.util.ArrayList;
import java.util.List;

public final class Duration
implements TemporalAmount,
Comparable<Duration>,
Serializable {
    static final long LONG_MASK = 0xFFFFFFFFL;
    public static final Duration ZERO = new Duration(0L, 0);
    private final long seconds;
    private final int nanos;

    public static Duration ofDays(long days) {
        return Duration.create(days * 86400L, 0);
    }

    public static Duration ofHours(long hours) {
        return Duration.create(hours * 3600L, 0);
    }

    public static Duration ofMinutes(long minutes) {
        return Duration.create(minutes * 60L, 0);
    }

    public static Duration ofSeconds(long seconds) {
        return Duration.create(seconds, 0);
    }

    public static Duration ofSeconds(long seconds, long nanoAdjustment) {
        long secs = seconds + MathUtils.floorDiv(nanoAdjustment, 1000000000L);
        int nos = (int)MathUtils.floorMod(nanoAdjustment, 1000000000L);
        return Duration.create(secs, nos);
    }

    public static Duration ofMillis(long millis) {
        long secs = millis / 1000L;
        int mos = (int)(millis % 1000L);
        if (mos < 0) {
            mos += 1000;
            --secs;
        }
        return Duration.create(secs, mos * 1000000);
    }

    public static Duration ofNanos(long nanos) {
        long secs = nanos / 1000000000L;
        int nos = (int)(nanos % 1000000000L);
        if (nos < 0) {
            nos = (int)((long)nos + 1000000000L);
            --secs;
        }
        return Duration.create(secs, nos);
    }

    public static Duration of(long amount, TemporalUnit unit) {
        return ZERO.plus(amount, unit);
    }

    public static Duration from(TemporalAmount amount) {
        Duration duration = ZERO;
        for (TemporalUnit unit : amount.getUnits()) {
            assert (unit != null);
            duration = duration.plus(amount.get(unit), unit);
        }
        return duration;
    }

    public static Duration between(Temporal startInclusive, Temporal endExclusive) {
        try {
            return Duration.ofNanos(startInclusive.until(endExclusive, ChronoUnit.NANOS));
        }
        catch (DateTimeException dateTimeException) {
            long nanos;
            long secs = startInclusive.until(endExclusive, ChronoUnit.SECONDS);
            try {
                nanos = endExclusive.getLong(ChronoField.NANO_OF_SECOND) - startInclusive.getLong(ChronoField.NANO_OF_SECOND);
                if (secs > 0L && nanos < 0L) {
                    ++secs;
                } else if (secs < 0L && nanos > 0L) {
                    --secs;
                }
            }
            catch (DateTimeException dateTimeException2) {
                nanos = 0L;
            }
            return Duration.ofSeconds(secs, nanos);
        }
    }

    private static Duration create(long seconds, int nanoAdjustment) {
        if ((seconds | (long)nanoAdjustment) == 0L) {
            return ZERO;
        }
        return new Duration(seconds, nanoAdjustment);
    }

    private Duration(long seconds, int nanos) {
        this.seconds = seconds;
        this.nanos = nanos;
    }

    @Override
    public long get(TemporalUnit unit) {
        if (unit == ChronoUnit.SECONDS) {
            return this.seconds;
        }
        if (unit == ChronoUnit.NANOS) {
            return this.nanos;
        }
        throw new UnsupportedTemporalTypeException("Unsupported unit: " + unit);
    }

    @Override
    public List<TemporalUnit> getUnits() {
        ArrayList<TemporalUnit> units = new ArrayList<TemporalUnit>(2);
        units.add(ChronoUnit.SECONDS);
        units.add(ChronoUnit.NANOS);
        return units;
    }

    public boolean isZero() {
        return (this.seconds | (long)this.nanos) == 0L;
    }

    public boolean isNegative() {
        return this.seconds < 0L;
    }

    public long getSeconds() {
        return this.seconds;
    }

    public int getNano() {
        return this.nanos;
    }

    public Duration withSeconds(long seconds) {
        return Duration.create(seconds, this.nanos);
    }

    public Duration withNanos(int nanoOfSecond) {
        ChronoField.NANO_OF_SECOND.checkValidIntValue(nanoOfSecond);
        return Duration.create(this.seconds, nanoOfSecond);
    }

    public Duration plus(Duration duration) {
        return this.plus(duration.getSeconds(), duration.getNano());
    }

    public Duration plus(long amountToAdd, TemporalUnit unit) {
        if (unit == ChronoUnit.DAYS) {
            return this.plus(amountToAdd * 86400L, 0L);
        }
        if (unit.isDurationEstimated()) {
            throw new UnsupportedTemporalTypeException("Unit must not have an estimated duration");
        }
        if (amountToAdd == 0L) {
            return this;
        }
        if (unit instanceof ChronoUnit) {
            switch ((ChronoUnit)unit) {
                case NANOS: {
                    return this.plusNanos(amountToAdd);
                }
                case MICROS: {
                    return this.plusSeconds(amountToAdd / 1000000000L * 1000L).plusNanos(amountToAdd % 1000000000L * 1000L);
                }
                case MILLIS: {
                    return this.plusMillis(amountToAdd);
                }
                case SECONDS: {
                    return this.plusSeconds(amountToAdd);
                }
            }
            return this.plusSeconds(unit.getDuration().seconds * amountToAdd);
        }
        Duration duration = unit.getDuration().multipliedBy(amountToAdd);
        return this.plusSeconds(duration.getSeconds()).plusNanos(duration.getNano());
    }

    public Duration plusDays(long daysToAdd) {
        return this.plus(daysToAdd * 86400L, 0L);
    }

    public Duration plusHours(long hoursToAdd) {
        return this.plus(hoursToAdd * 3600L, 0L);
    }

    public Duration plusMinutes(long minutesToAdd) {
        return this.plus(minutesToAdd * 60L, 0L);
    }

    public Duration plusSeconds(long secondsToAdd) {
        return this.plus(secondsToAdd, 0L);
    }

    public Duration plusMillis(long millisToAdd) {
        return this.plus(millisToAdd / 1000L, millisToAdd % 1000L * 1000000L);
    }

    public Duration plusNanos(long nanosToAdd) {
        return this.plus(0L, nanosToAdd);
    }

    private Duration plus(long secondsToAdd, long nanosToAdd) {
        if ((secondsToAdd | nanosToAdd) == 0L) {
            return this;
        }
        long epochSec = this.seconds + secondsToAdd;
        epochSec += nanosToAdd / 1000000000L;
        long nanoAdjustment = (long)this.nanos + (nanosToAdd %= 1000000000L);
        return Duration.ofSeconds(epochSec, nanoAdjustment);
    }

    public Duration minus(Duration duration) {
        long secsToSubtract = duration.getSeconds();
        int nanosToSubtract = duration.getNano();
        if (secsToSubtract == Long.MIN_VALUE) {
            return this.plus(Long.MAX_VALUE, -nanosToSubtract).plus(1L, 0L);
        }
        return this.plus(-secsToSubtract, -nanosToSubtract);
    }

    public Duration minus(long amountToSubtract, TemporalUnit unit) {
        return amountToSubtract == Long.MIN_VALUE ? this.plus(Long.MAX_VALUE, unit).plus(1L, unit) : this.plus(-amountToSubtract, unit);
    }

    public Duration minusDays(long daysToSubtract) {
        return daysToSubtract == Long.MIN_VALUE ? this.plusDays(Long.MAX_VALUE).plusDays(1L) : this.plusDays(-daysToSubtract);
    }

    public Duration minusHours(long hoursToSubtract) {
        return hoursToSubtract == Long.MIN_VALUE ? this.plusHours(Long.MAX_VALUE).plusHours(1L) : this.plusHours(-hoursToSubtract);
    }

    public Duration minusMinutes(long minutesToSubtract) {
        return minutesToSubtract == Long.MIN_VALUE ? this.plusMinutes(Long.MAX_VALUE).plusMinutes(1L) : this.plusMinutes(-minutesToSubtract);
    }

    public Duration minusSeconds(long secondsToSubtract) {
        return secondsToSubtract == Long.MIN_VALUE ? this.plusSeconds(Long.MAX_VALUE).plusSeconds(1L) : this.plusSeconds(-secondsToSubtract);
    }

    public Duration minusMillis(long millisToSubtract) {
        return millisToSubtract == Long.MIN_VALUE ? this.plusMillis(Long.MAX_VALUE).plusMillis(1L) : this.plusMillis(-millisToSubtract);
    }

    public Duration minusNanos(long nanosToSubtract) {
        return nanosToSubtract == Long.MIN_VALUE ? this.plusNanos(Long.MAX_VALUE).plusNanos(1L) : this.plusNanos(-nanosToSubtract);
    }

    public Duration multipliedBy(long multiplicand) {
        if (multiplicand == 0L) {
            return ZERO;
        }
        if (multiplicand == 1L) {
            return this;
        }
        return Duration.create(this.toNanosInternal().multiply(multiplicand));
    }

    public Duration dividedBy(long divisor) {
        if (divisor == 0L) {
            throw new ArithmeticException("Cannot divide by zero");
        }
        if (divisor == 1L) {
            return this;
        }
        return Duration.create(this.toNanosInternal().divide(BigInteger.valueOf(divisor)));
    }

    private BigInteger toNanosInternal() {
        return BigInteger.valueOf(this.seconds).multiply(1000000000L).add(BigInteger.valueOf(this.nanos));
    }

    private static Duration create(BigInteger nanos) {
        BigInteger[] divRem = nanos.divideAndRemainder(BigInteger.valueOf(1000000000L));
        if (divRem[0].bitLength() > 63) {
            throw new ArithmeticException("Exceeds capacity of Duration: " + nanos);
        }
        return Duration.ofSeconds(divRem[0].longValue(), divRem[1].intValue());
    }

    public Duration negated() {
        return this.multipliedBy(-1L);
    }

    public Duration abs() {
        return this.isNegative() ? this.negated() : this;
    }

    @Override
    public Temporal addTo(Temporal temporal) {
        if (this.seconds != 0L) {
            temporal = temporal.plus(this.seconds, ChronoUnit.SECONDS);
        }
        if (this.nanos != 0) {
            temporal = temporal.plus(this.nanos, ChronoUnit.NANOS);
        }
        return temporal;
    }

    @Override
    public Temporal subtractFrom(Temporal temporal) {
        if (this.seconds != 0L) {
            temporal = temporal.minus(this.seconds, ChronoUnit.SECONDS);
        }
        if (this.nanos != 0) {
            temporal = temporal.minus(this.nanos, ChronoUnit.NANOS);
        }
        return temporal;
    }

    public long toDays() {
        return this.seconds / 86400L;
    }

    public long toHours() {
        return this.seconds / 3600L;
    }

    public long toMinutes() {
        return this.seconds / 60L;
    }

    public long toMillis() {
        long millis = this.seconds * 1000L;
        return millis += (long)(this.nanos / 1000000);
    }

    public long toNanos() {
        long totalNanos = this.seconds * 1000000000L;
        return totalNanos += (long)this.nanos;
    }

    @Override
    public int compareTo(Duration otherDuration) {
        int cmp = Long.compare(this.seconds, otherDuration.seconds);
        if (cmp != 0) {
            return cmp;
        }
        return this.nanos - otherDuration.nanos;
    }

    public boolean equals(@Nullable Object otherDuration) {
        if (this == otherDuration) {
            return true;
        }
        if (otherDuration instanceof Duration) {
            Duration other = (Duration)otherDuration;
            return this.seconds == other.seconds && this.nanos == other.nanos;
        }
        return false;
    }

    public int hashCode() {
        return (int)(this.seconds ^ this.seconds >>> 32) + 51 * this.nanos;
    }

    public String toString() {
        if (this == ZERO) {
            return "PT0S";
        }
        long hours = this.seconds / 3600L;
        int minutes = (int)(this.seconds % 3600L / 60L);
        int secs = (int)(this.seconds % 60L);
        StringBuilder buf = new StringBuilder(24);
        buf.append("PT");
        if (hours != 0L) {
            buf.append(hours).append('H');
        }
        if (minutes != 0) {
            buf.append(minutes).append('M');
        }
        if (secs == 0 && this.nanos == 0 && buf.length() > 2) {
            return buf.toString();
        }
        if (secs < 0 && this.nanos > 0) {
            if (secs == -1) {
                buf.append("-0");
            } else {
                buf.append(secs + 1);
            }
        } else {
            buf.append(secs);
        }
        if (this.nanos > 0) {
            int pos = buf.length();
            if (secs < 0) {
                buf.append(2000000000L - (long)this.nanos);
            } else {
                buf.append((long)this.nanos + 1000000000L);
            }
            while (buf.charAt(buf.length() - 1) == '0') {
                buf.setLength(buf.length() - 1);
            }
            buf.setCharAt(pos, '.');
        }
        buf.append('S');
        return buf.toString();
    }
}

