/*
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.audio;

import java.io.Closeable;

/**
 * Represents an audio playback stream.
 * <p>
 * An audio track allocates a native buffer when it is opened. The audio track should be closed with the
 * {@link #close()} method in order to free the native allocation.
 * <p>
 * While the audio track is open, the native implementation reads audio data continuously from the buffer and plays it
 * on the output device. The {@link #writeBuffer(byte[], int, int)} method can be used to write a chunk of audio data in
 * the buffer. If audio data is not written fast enough by the application, the output device may play undesired
 * silences. The {@link #waitForBufferFlushed()} method can be used to wait until all the audio data written in the
 * buffer has been played back.
 * <p>
 * The volume of the playback can be configured by calling {@link #setVolume(float)}.
 */
public class AudioTrack implements Closeable {

	/**
	 * Creates and opens an audio track.
	 *
	 * @param format
	 *            the format.
	 * @param bufferSize
	 *            the buffer size (in bytes).
	 * @throws IllegalArgumentException
	 *             if the given buffer size is less than or equal to zero.
	 * @throws AudioException
	 *             if the audio track could not be opened.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an audio track.
	 */
	public AudioTrack(AudioFormat format, int bufferSize) {
		throw new RuntimeException();
	}

	/**
	 * Returns the format of this audio track.
	 *
	 * @return the format.
	 */
	public AudioFormat getFormat() {
		throw new RuntimeException();
	}

	/**
	 * Returns the buffer size of this audio track.
	 *
	 * @return the buffer size (in bytes).
	 */
	public int getBufferSize() {
		throw new RuntimeException();
	}

	/**
	 * Returns whether this audio track is closed.
	 *
	 * @return whether this audio track is closed.
	 */
	public boolean isClosed() {
		throw new RuntimeException();
	}

	/**
	 * Closes this audio track.
	 * <p>
	 * This method releases the native resources allocated when opening this audio track. This method does nothing if
	 * this audio track is already closed.
	 */
	@Override
	public void close() {
		throw new RuntimeException();
	}

	/**
	 * Sets the volume of this audio track.
	 *
	 * @param gain
	 *            the gain (between <code>0.0f</code> and <code>1.0f</code>).
	 */
	public void setVolume(float gain) {
		throw new RuntimeException();
	}

	/**
	 * Writes audio data in the buffer of this audio track.
	 * <p>
	 * This method blocks until the requested size has been written, until this audio track is closed or until the
	 * thread is interrupted.
	 * <p>
	 * The number of bytes written may be different from the given size if this audio track is closed or until the
	 * thread is interrupted.
	 *
	 * @param array
	 *            the array containing the data to write.
	 * @param offset
	 *            the offset in the array of the first byte to write.
	 * @param size
	 *            the maximum number of bytes to write.
	 * @return the number of bytes written.
	 * @throws IndexOutOfBoundsException
	 *             if the given offset or the given size is invalid.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to write in the buffer.
	 */
	public int writeBuffer(byte[] array, int offset, int size) {
		throw new RuntimeException();
	}

	/**
	 * Waits for the buffer of this audio track to be flushed.
	 * <p>
	 * This method blocks until the buffer is empty (until all the audio data has been played back), until this audio
	 * track is closed or until the thread is interrupted.
	 *
	 * @return <code>true</code> if the buffer is empty, <code>false</code> if this audio track is closed or the thread
	 *         is interrupted.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to wait for the buffer.
	 */
	public boolean waitForBufferFlushed() {
		throw new RuntimeException();
	}
}
