/*
 * Java
 *
 * Copyright 2011-2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.sp;

/**
 * <p>
 * A shielded plug is a database that contains several memory blocks.
 * </p>
 * <p>
 * A shielded plug can be created at runtime using {@link #createDatabase(int)} or be created at startup and retrieved
 * by {@link #getDatabase(int)}.
 * </p>
 * <p>
 * The list of memory blocks IDs can be retrieve using {@link #getIDs()}.<br>
 * There are two sorts of shielded plugs:
 * </p>
 * <ol>
 * <li>The immutable ones ({@link #isImmutable()}) that cannot be modified.
 * <li>The mutable ones (!{@link #isImmutable()}) can be modified by adding or removing blocks using
 * {@link #create(int, int)} or {@link #create(int, int, int)} or {@link #delete(int)}.
 * </ol>
 * <p>
 * Each block has fixed length ({@link #getLength(int)} passing the block ID).<br>
 * All access to a database is serialized by the implementation: there will be only one access (either read or write) at
 * a time. Each access to a block is atomic, this avoids inconsistency:
 * </p>
 * <ul>
 * <li>It can be read using one of the read methods that match its length.</li>
 * <li>It can be written using one the write methods that match its length.</li>
 * </ul>
 * <p>
 * Each memory block has a flag that indicates that an update has occurred since the last read. It is possible to test
 * this state: {@link #isPending(int)}. This flag is set to false when reading, and to true when writing.
 * </p>
 * <p>
 * A task can wait for the modification of a memory block by using {@link #waitFor(int)}. This method suspends the
 * current task if and only if the method pending returns false on the specified memory block. A task can also wait on
 * several memory blocks, the task is released when one of the blocks is modified {@link #waitFor(int[])}).
 * </p>
 * <p>
 * Each memory block has a flag indicating if its data are available or not. It is possible to test this flag using
 * {@link #isDataAvailable(int)}. This flag is initially <code>false</code> and is set to <code>true</code> when writing
 * data. It can be set to <code>false</code> using the method {@link #reset(int)}.
 * </p>
 */
public class ShieldedPlug {

	/**
	 * Could not create a <code>ShieldedPlug</code>
	 */
	private ShieldedPlug() {
		// no constructor available
		throw new RuntimeException();
	}

	/**
	 * Returns the database defined at the given ID.
	 *
	 * @param ID
	 *            the identification number of the requested database
	 * @return the database with the given ID
	 * @throws IllegalArgumentException
	 *             if no database is defined with the given ID
	 */
	public static ShieldedPlug getDatabase(int ID) {
		throw new RuntimeException();
	}

	/**
	 * Creates a new empty database with the given ID.
	 *
	 * @param ID
	 *            the identification number of the created database
	 * @return the created database
	 * @throws IllegalArgumentException
	 *             if a database with the given ID already exists
	 * @throws SecurityException
	 *             if the platform cannot create dynamically databases
	 */
	public static ShieldedPlug createDatabase(int ID) {
		throw new RuntimeException();
	}

	/**
	 * Gets if this database is immutable or not.
	 *
	 * @return <code>true</code> if no block can be added or remove to this database, <code>false</code> otherwise
	 */
	public boolean isImmutable() {
		throw new RuntimeException();
	}

	/**
	 * Deletes the block with the given ID.
	 *
	 * @param blockID
	 *            the ID of the block to delete
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws SecurityException
	 *             if this database is immutable
	 */
	public void delete(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Creates a block with the given ID.
	 *
	 * @param blockID
	 *            the ID of the block to create
	 * @param length
	 *            the length in bytes of the block to create
	 * @param maxTasks
	 *            maximum number of tasks that can wait at the same time for a modification of the block
	 * @throws IllegalArgumentException
	 *             if a block is already defined with the given ID
	 * @throws SecurityException
	 *             if this database is immutable
	 */
	public void create(int blockID, int length, int maxTasks) {
		throw new RuntimeException();
	}

	/**
	 * Creates a block with the given ID. An unlimited number of tasks will be able to wait at the same time for a
	 * modification of the block.
	 *
	 * @param blockID
	 *            the ID of the block to create
	 * @param length
	 *            the length in bytes of the block to create
	 * @throws IllegalArgumentException
	 *             if a block is already defined with the given ID
	 * @throws SecurityException
	 *             if this database is immutable
	 */
	public void create(int blockID, int length) {
		throw new RuntimeException();
	}

	/**
	 * Gets the ID of this database.
	 *
	 * @return the ID of this database
	 */
	public int getID() {
		throw new RuntimeException();
	}

	/**
	 * Gets the number of blocks of this database.
	 *
	 * @return the number of blocks in this database
	 */
	public int getSize() {
		throw new RuntimeException();
	}

	/**
	 * Gets the list of IDs of the blocks available in this database.
	 *
	 * @return the list of the IDs of the blocks available in this database
	 */
	public int[] getIDs() {
		throw new RuntimeException();
	}

	/**
	 * Returns the length of the block with the given ID.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the length in bytes
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public int getLength(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Gets the maximum number of tasks that can wait at the same time on the block defined with the given ID.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the maximum number of tasks that can wait at the same time on the block defined with the given ID, or -1
	 *         if infinite
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public int getMaxTasks(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Fills the given array with data from the block with the given ID. The number of bytes read is equal to the length
	 * of the block.<br>
	 * The <code>read(blockID, data)</code> method has the same effect as:
	 * 
	 * <pre>
	 * <code> read(blockID, data, 0) </code>
	 * </pre>
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param data
	 *            the buffer into which the data is read
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if <code>data.length</code> is lower than block length
	 */
	public void read(int blockID, byte[] data) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Fills the given array with <code>block.length</code> bytes from the block with the given ID. The first byte read
	 * is stored into element <code>data[destOffset]</code>.
	 * <p>
	 * If <code>destOffset</code> is negative or <code>destOffset</code> + block length is greater than the length of
	 * the array <code>data</code>, then an <code>IndexOutOfBoundsException</code> is thrown.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param data
	 *            the buffer into which the data is read
	 * @param destOffset
	 *            the start offset in array <code>data</code> at which the data is written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if <code>destOffset</code> is negative or if <code>data.length</code> is lower than
	 *             <code>destOffset</code> + block length
	 */
	public void read(int blockID, byte[] data, int destOffset) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Reads four input bytes from the block with the given ID and returns an <code>int</code> value.<br>
	 * The way the <code>int</code> is built from the four bytes is platform dependent.<br>
	 * <p>
	 * This method is suitable for reading bytes written by the <code>writeInt</code> method.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the <code>int</code> value read
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is not four bytes
	 */
	public int readInt(int blockID) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Reads eight input bytes from the block with the given ID and returns a <code>long</code> value.<br>
	 * The way the <code>long</code> is built from the eight bytes is platform dependent.<br>
	 * <p>
	 * This method is suitable for reading bytes written by the <code>writeLong</code> method.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the <code>long</code> value read
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is not height bytes
	 */
	public long readLong(int blockID) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Reads four input bytes from the block with the given ID and returns a <code>float</code> value.<br>
	 * The way the <code>float</code> is built from the four bytes is platform dependent.<br>
	 * <p>
	 * This method is suitable for reading bytes written by the <code>writeFloat</code> method.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the <code>float</code> value read
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is not four bytes
	 */
	public float readFloat(int blockID) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Reads eight input bytes from the block with the given ID and returns a <code>double</code> value.<br>
	 * The way the <code>double</code> is built from the eight bytes is platform dependent.<br>
	 * <p>
	 * This method is suitable for reading bytes written by the <code>writeDouble</code> method.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the <code>double</code> value read
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is not height bytes
	 */
	public double readDouble(int blockID) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Invokes the <code>readObject</code> method of the {@link SPReader} registered for the block with the given ID.
	 * <br>
	 * The {@link SPReader} is responsible for the de-serialization of the object from the block.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the object read from the block
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws EmptyBlockException
	 *             if no data is available in the block
	 * @throws NullPointerException
	 *             if no {@link SPReader} has been registered for the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is lower than the size needed for object de-serialization
	 */
	public Object readObject(int blockID) throws EmptyBlockException {
		throw new RuntimeException();
	}

	/**
	 * Registers the given {@link SPReader} to de-serialize objects from the block with the given ID.<br>
	 * If an {@link SPReader} is already defined for the block, it is replaced by the given {@link SPReader}.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param reader
	 *            the {@link SPReader}
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public void setReader(int blockID, SPReader reader) {
		throw new RuntimeException();
	}

	/**
	 * Gets the {@link SPReader} used to de-serialize objects from the block with the given ID.<br>
	 * If no {@link SPReader} is defined for the block, <code>null</code> is returned.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the {@link SPReader} set or null if none
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public SPReader getReader(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Writes block length bytes from the specified byte array to the block with the given ID.<br>
	 * The <code>write(blockID, data)</code> method has the same effect as:
	 * 
	 * <pre>
	 * <code> write(blockID, data, 0) </code>
	 * </pre>
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param data
	 *            the data to write
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if <code>data.length</code> value is lower than block length
	 */
	public void write(int blockID, byte[] data) {
		throw new RuntimeException();
	}

	/**
	 * Writes block length bytes from the specified byte array to the block with the given ID.<br>
	 * Element <code>data[destOffset]</code> is the first byte written to the block.
	 * <p>
	 * If <code>destOffset</code> is negative, or <code>destOffset</code> + block length is greater than the length of
	 * the array <code>data</code>, then an <code>IndexOutOfBoundsException</code> is thrown.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param data
	 *            the data to write
	 * @param srcOffset
	 *            the start offset in the data
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if <code>destOffset</code> is negative or if <code>data.length</code> is lower than
	 *             <code>offset</code> + block length
	 */
	public void write(int blockID, byte[] data, int srcOffset) {
		throw new RuntimeException();
	}

	/**
	 * Writes an <code>int</code> value, which is comprised of four bytes, to the block with the given ID.<br>
	 * The way the <code>int</code> is written from the four bytes is platform dependent.<br>
	 * <p>
	 * The bytes written by this method may be read by the <code>readInt</code> method, which will then return an
	 * <code>int</code> equal to <code>value</code>.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param value
	 *            the <code>int</code> value to be written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if block length is not four bytes
	 */
	public void writeInt(int blockID, int value) {
		throw new RuntimeException();
	}

	/**
	 * Writes a <code>long</code> value, which is comprised of eight bytes, to the block with the given ID.<br>
	 * The way the <code>long</code> is written from the eight bytes is platform dependent.<br>
	 * <p>
	 * The bytes written by this method may be read by the <code>readLong</code> method, which will then return a
	 * <code>long</code> equal to <code>value</code>.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param value
	 *            the <code>long</code> value to be written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if block length is not eight bytes
	 */
	public void writeLong(int blockID, long value) {
		throw new RuntimeException();
	}

	/**
	 * Writes a <code>float</code> value, which is comprised of four bytes, to the block with the given ID.<br>
	 * The way the <code>float</code> is written from the four bytes is platform dependent.<br>
	 * <p>
	 * The bytes written by this method may be read by the <code>readFloat</code> method, which will then return a
	 * <code>float</code> equal to <code>value</code>.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param value
	 *            the <code>float</code> value to be written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if block length is not four bytes
	 */
	public void writeFloat(int blockID, float value) {
		throw new RuntimeException();
	}

	/**
	 * Writes a <code>double</code> value, which is comprised of eight bytes, to the block with the given ID.<br>
	 * The way the <code>double</code> is written from the eight bytes is platform dependent.<br>
	 * <p>
	 * The bytes written by this method may be read by the <code>readDouble</code> method, which will then return a
	 * <code>double</code> equal to <code>value</code>.
	 *
	 * @param blockID
	 *            ID of the block
	 * @param value
	 *            the <code>double</code> value to be written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws IndexOutOfBoundsException
	 *             if block length is not eight bytes
	 */
	public void writeDouble(int blockID, double value) {
		throw new RuntimeException();
	}

	/**
	 * Invokes the <code>writeObject</code> method of the {@link SPWriter} registered for the block with the given ID.
	 * <br>
	 * The {@link SPWriter} is responsible for the serialization of the object into the block.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param o
	 *            the object to be written
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws NullPointerException
	 *             if no {@link SPWriter} has been registered for the block
	 * @throws IndexOutOfBoundsException
	 *             if block length is lower than the size needed for object serialization
	 */
	public void writeObject(int blockID, Object o) {
		throw new RuntimeException();
	}

	/**
	 * Registers the given {@link SPWriter} to serialize objects into the block with the given ID.<br>
	 * If an {@link SPWriter} is already defined for the block, it is replaced by the given {@link SPWriter}.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @param writer
	 *            the {@link SPWriter}
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public void setWriter(int blockID, SPWriter writer) {
		throw new RuntimeException();
	}

	/**
	 * Gets the {@link SPWriter} used to serialize objects into the block with the given ID.<br>
	 * If no {@link SPWriter} is defined for the block, <code>null</code> is returned.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return the {@link SPWriter} set or null if none
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public SPWriter getWriter(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Causes current thread to wait until another thread write data into the block with the given ID.<br>
	 * If data has been written in the block since last read, this method returns immediately.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @throws InterruptedException
	 *             if another thread has interrupted the current thread The interrupted status of the current thread is
	 *             cleared when this exception is thrown
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 * @throws TooManyWaitingThreadsException
	 *             if too many threads are waiting for new data
	 */
	public void waitFor(int blockID) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * Causes current thread to wait until another thread write data into at least one block from the blocks with the
	 * given IDs.<br>
	 * If data has been written in one block since last read from it, this method returns immediately.
	 *
	 * @param blockIDs
	 *            the list of block IDs
	 * @return the list of IDs of the blocks that has been written
	 * @throws InterruptedException
	 *             if another thread has interrupted the current thread The interrupted status of the current thread is
	 *             cleared when this exception is thrown
	 * @throws IllegalArgumentException
	 *             if one of the ID does not correspond to an existing block
	 * @throws TooManyWaitingThreadsException
	 *             if too many threads are waiting for new data
	 */
	public int[] waitFor(int[] blockIDs) throws InterruptedException {
		throw new RuntimeException();
	}

	/**
	 * Gets if data has been written into the block since last read.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return <code>true</code> if data has been written into the block since last read, <code>false</code> otherwise
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public boolean isPending(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Determines whether data in the block with the given ID are available or not.<br>
	 * By default no data is available in a block. When data are written in a block, they remain available until method
	 * <code>reset(int)</code> is called.
	 *
	 * @param blockID
	 *            the ID of the block
	 * @return <code>true</code> if data is available in the block <code>false</code> otherwise
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public boolean isDataAvailable(int blockID) {
		throw new RuntimeException();
	}

	/**
	 * Resets data of the block with the given ID.<br>
	 * After execution of this method, {@link #isDataAvailable(int)} method would return <code>false</code> (unless data
	 * were written after calling <code>reset(int)</code> and before calling {@link #isDataAvailable(int)}).
	 *
	 * @param blockID
	 *            the ID of the block
	 * @throws IllegalArgumentException
	 *             if no block is defined with the given ID
	 */
	public void reset(int blockID) {
		throw new RuntimeException();
	}
}
