/*
 * Java
 *
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

import ej.microui.display.LLUIPainter.MicroUIGraphicsContext;
import ej.microui.display.LLUIPainter.MicroUIImage;

/**
 * This interface provides all drawing methods useful to implement MicroUI drawing native methods.
 * <p>
 * This interface is implemented by the MicroUI graphical engine's internal software drawer. This drawer implements all
 * drawings in software. The rendering is exactly the same than the embedded algorithms listed in
 * <code>ui_drawing_soft.h</code>. See {@link LLUIDisplay#getUIDrawerSoftware()}.
 * <p>
 * This interface can be fully implemented by a dedicated platform class in platform front panel project. This allows to
 * implement drawing algorithms like embedded platform.
 * <p>
 * Additionally, this interface can be only partially implemented by a dedicated platform class in platform front panel
 * project. This class must implement {@link UIDrawingDefault} interface instead and override only a set of drawing
 * methods. This allows to implement same drawing algorithms like embedded platform. All others drawing algorithms are
 * using the MicroUI graphical engine's internal software drawer.
 */
public interface UIDrawing {

	/**
	 * Default implementation of {@link UIDrawing} which uses the MicroUI graphical engine's internal software drawer to
	 * perform all drawings. This interface can be implemented to override only a set of drawing methods which matches
	 * with the embedded implementation.
	 */
	public interface UIDrawingDefault extends UIDrawing {

		@Override
		default void writePixel(MicroUIGraphicsContext gc, int x, int y) {
			LLUIDisplay.Instance.getUIDrawerSoftware().writePixel(gc, x, y);
		}

		@Override
		default void drawLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawLine(gc, x1, y1, x2, y2);
		}

		@Override
		default void drawHorizontalLine(MicroUIGraphicsContext gc, int x1, int x2, int y) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawHorizontalLine(gc, x1, x2, y);
		}

		@Override
		default void drawVerticalLine(MicroUIGraphicsContext gc, int x, int y1, int y2) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawVerticalLine(gc, x, y1, y2);
		}

		@Override
		default void drawRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawRectangle(gc, x1, y1, x2, y2);
		}

		@Override
		default void fillRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillRectangle(gc, x, y, width, height);
		}

		@Override
		default void drawRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
				int arcHeight) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
		}

		@Override
		default void fillRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
				int arcHeight) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillRoundedRectangle(gc, x, y, width, height, arcWidth, arcHeight);
		}

		@Override
		default void drawCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
				float arcAngle) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawCircleArc(gc, x, y, diameter, startAngle, arcAngle);
		}

		@Override
		default void drawEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
				float arcAngle) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
		}

		@Override
		default void fillCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle,
				float arcAngle) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillCircleArc(gc, x, y, diameter, startAngle, arcAngle);
		}

		@Override
		default void fillEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
				float arcAngle) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillEllipseArc(gc, x, y, width, height, startAngle, arcAngle);
		}

		@Override
		default void drawEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawEllipse(gc, x, y, width, height);
		}

		@Override
		default void fillEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillEllipse(gc, x, y, width, height);
		}

		@Override
		default void drawCircle(MicroUIGraphicsContext gc, int x, int y, int diameter) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawCircle(gc, x, y, diameter);
		}

		@Override
		default void fillCircle(MicroUIGraphicsContext gc, int x, int y, int diameter) {
			LLUIDisplay.Instance.getUIDrawerSoftware().fillCircle(gc, x, y, diameter);
		}

		@Override
		default void drawImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width,
				int height, int x, int y, int alpha) {
			LLUIDisplay.Instance.getUIDrawerSoftware().drawImage(gc, img, regionX, regionY, width, height, x, y, alpha);
		}
	}

	/**
	 * Draws a pixel at given position.
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x
	 *            the pixel X coordinate.
	 * @param y
	 *            the pixel Y coordinate.
	 */
	void writePixel(MicroUIGraphicsContext gc, int x, int y);

	/**
	 * Draws a line at between points x1,y1 (included) and x2,y2 (included). Note: x1 may be higher than x2 (and/or y1
	 * may be higher than y2).
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	void drawLine(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2);

	/**
	 * Draws a horizontal line at between points x1,y (included) and x2,y (included). Caller ensures that x1 <= x2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the first pixel line X coordinate.
	 * @param x2
	 *            the last pixel line X coordinate.
	 * @param y
	 *            the both pixels line Y coordinate.
	 */
	void drawHorizontalLine(MicroUIGraphicsContext gc, int x1, int x2, int y);

	/**
	 * Draws a vertical line at between points x,y1 (included) and x,y2 (included). Caller ensures that y1 <= y2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the both pixels line X coordinate.
	 * @param y1
	 *            the first pixel line Y coordinate.
	 * @param y2
	 *            the last pixel line Y coordinate.
	 */
	void drawVerticalLine(MicroUIGraphicsContext gc, int x, int y1, int y2);

	/**
	 * Draws a rectangle at from top-left point x1,y1 (included) and bottom-right point x2,y2 (included). Caller ensures
	 * that x1 <= x2 and y1 <= y2.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x1
	 *            the top-left pixel X coordinate.
	 * @param y1
	 *            the top-left pixel Y coordinate.
	 * @param x2
	 *            the bottom-right pixel X coordinate.
	 * @param y2
	 *            the top-right pixel Y coordinate.
	 */
	void drawRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2);

	/**
	 * Fills a rectangle at from top-left point x1,y1 (included) and bottom-right point x2,y2 (included). Caller ensures
	 * that x1 <= x2 and y1 <= y2.
	 *
	 * @param gc
	 *            drawing destination target.
	 * @param x1
	 *            the top-left pixel X coordinate.
	 * @param y1
	 *            the top-left pixel Y coordinate.
	 * @param x2
	 *            the bottom-right pixel X coordinate.
	 * @param y2
	 *            the top-right pixel Y coordinate.
	 */
	void fillRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2);

	/**
	 * Draws a rounded rectangle at from top-left point x,y (included) and bottom-right point x+width-1,y+height-1
	 * (included).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	void drawRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
			int arcHeight);

	/**
	 * Fills a rounded rectangle at from top-left point x,y (included) and bottom-right point x+width-1,y+height-1
	 * (included).
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param arcWidth
	 *            the horizontal diameter of the arc at the corners.
	 * @param arcHeight
	 *            the vertical diameter of the arc at the corners.
	 */
	void fillRoundedRectangle(MicroUIGraphicsContext gc, int x, int y, int width, int height, int arcWidth,
			int arcHeight);

	/**
	 * Draws a circular arc covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by diameter.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	void drawCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle);

	/**
	 * Draws an elliptical arc covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included) .
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	void drawEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle);

	/**
	 * Fills a circular arc covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 *
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * square whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 *
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 *
	 * If either the given diameter is negative or zero, or if arcAngle is zero, nothing is drawn.
	 *
	 * The angles are given relative to the square. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the square to the upper right corner of the square.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the diameter of the arc to draw.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	void fillCircleArc(MicroUIGraphicsContext gc, int x, int y, int diameter, float startAngle, float arcAngle);

	/**
	 * Fills an arc covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * The arc is drawn from startAngle up to arcAngle degrees. The center of the arc is defined as the center of the
	 * rectangle whose origin is at (x,y) (upper-left corner) and whose dimension is given by width and height.
	 * <p>
	 * Angles are interpreted such that 0 degrees is at the 3 o'clock position. A positive value indicates a
	 * counter-clockwise rotation while a negative value indicates a clockwise rotation.
	 * <p>
	 * If either the given width or height is negative or zero, or if arcAngle is zero, nothing is drawn.
	 * <p>
	 * The angles are given relative to the rectangle. For instance an angle of 45 degrees is always defined by the line
	 * from the center of the rectangle to the upper right corner of the rectangle. Thus for a non squared rectangle
	 * angles are skewed along either height or width.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the rectangle width.
	 * @param height
	 *            the rectangle height.
	 * @param startAngle
	 *            the beginning angle of the arc to draw
	 * @param arcAngle
	 *            the angular extent of the arc from startAngle
	 */
	void fillEllipseArc(MicroUIGraphicsContext gc, int x, int y, int width, int height, float startAngle,
			float arcAngle);

	/**
	 * Draws an ellipse covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	void drawEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height);

	/**
	 * Fills an ellipse covering the rectangle defined by top-left point x,y (included) and bottom-right point
	 * x+width-1,y+height-1 (included).
	 * <p>
	 * If either the given width or height is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param width
	 *            the ellipse width.
	 * @param height
	 *            the ellipse height.
	 */
	void fillEllipse(MicroUIGraphicsContext gc, int x, int y, int width, int height);

	/**
	 * Draws a circle covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	void drawCircle(MicroUIGraphicsContext gc, int x, int y, int diameter);

	/**
	 * Fills a circle covering the square defined by top-left point x,y (included) and bottom-right point
	 * x+diameter-1,y+diameter-1 (included).
	 * <p>
	 * If the given diameter is negative or zero, nothing is drawn.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param x
	 *            the top-left pixel X coordinate.
	 * @param y
	 *            the top-left pixel Y coordinate.
	 * @param diameter
	 *            the circle square size.
	 */
	void fillCircle(MicroUIGraphicsContext gc, int x, int y, int diameter);

	/**
	 * Draws a region of an image. The region of the image to draw is given relative to the image (origin at the
	 * upper-left corner) as a rectangle.
	 * <p>
	 * If the specified source region exceeds the image bounds, the copied region is limited to the image boundary. If
	 * the copied region goes out of the bounds of the graphics context area, pixels out of the range will not be drawn.
	 * <p>
	 * A global opacity value is given. When this value is 0xff (255, opaque), that means the image is drawn on the
	 * graphics context without managing an extra opacity. Only the image transparent pixels must have been merged with
	 * destination. All image opaque pixels override destination.
	 * <p>
	 * When this value is a value between 0 and 0xff, that means each pixel of the image must be merged with destination
	 * in addition with the image transparent pixels. An image opaque pixel becomes transparent (its opacity is the
	 * given alpha) and the opacity of an image transparent pixel becomes (alpha * alpha(pixel)) / 255.
	 *
	 * @param gc
	 *            the MicroUI GraphicsContext target.
	 * @param img
	 *            the MicroUI Image to draw.
	 * @param regionX
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param regionY
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @param x
	 *            the x coordinate of the top-left point in the destination.
	 * @param y
	 *            the y coordinate of the top-left point in the destination.
	 * @param alpha
	 *            the opacity level to apply to the region.
	 */
	void drawImage(MicroUIGraphicsContext gc, MicroUIImage img, int regionX, int regionY, int width, int height, int x,
			int y, int alpha);

}
