/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.hoka.http.encoding;

import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.hoka.http.HttpConstants;
import ej.hoka.http.support.AcceptEncoding;
import ej.hoka.http.support.QualityArgument;

/**
 * Class that stores a register of available encoding and transfer coding handlers.
 */
public class EncodingRegistry {

	private final PackedMap<String, ContentEncoding> encodingHandlers;
	private final PackedMap<String, TransferEncoding> transferCodingHandlers;

	/**
	 * Constructs the {@link EncodingRegistry} with {@link IdentityContentEncoding}, {@link IdentityTransferEncoding}
	 * and {@link ChunkedTransferCodingHandler} registered.
	 */
	public EncodingRegistry() {
		this.encodingHandlers = new PackedMap<>();
		this.transferCodingHandlers = new PackedMap<>();

		TransferEncoding chunkedTransferCodingHandler = ChunkedTransferCodingHandler.getInstance();
		this.transferCodingHandlers.put(chunkedTransferCodingHandler.getId().toLowerCase(),
				chunkedTransferCodingHandler);
	}

	/**
	 * Return the {@link ContentEncoding} corresponding to identity transfer coding (i.e. no transfer coding).
	 *
	 * @return Return the {@link ContentEncoding} corresponding to identity transfer coding (i.e. no transfer coding)
	 */
	public TransferEncoding getIdentityTransferCodingHandler() {
		return IdentityTransferEncoding.getInstance();
	}

	/**
	 * Return the {@link ContentEncoding} corresponding to chunked transfer coding.
	 *
	 * @return Return the {@link ContentEncoding} corresponding to chunked transfer coding.
	 */
	public TransferEncoding getChunkedTransferEncoding() {
		return ChunkedTransferCodingHandler.getInstance();
	}

	/**
	 * Return the {@link ContentEncoding} corresponding to the given encoding.
	 *
	 * @param encoding
	 *            case insensitive (See RFC2616, 3.5).
	 * @return null if no handler has been registered to match this encoding.
	 */
	@Nullable
	public ContentEncoding getContentEncoding(@Nullable String encoding) {
		if (encoding == null || encoding.isEmpty() || HttpConstants.WILDCARD.equals(encoding)) {
			return IdentityContentEncoding.getInstance();
		}

		return this.encodingHandlers.get(encoding.toLowerCase());
	}

	/**
	 * Return the {@link TransferEncoding} corresponding to the given encoding.
	 *
	 * @param encoding
	 *            case insensitive (See RFC2616, 3.5).
	 * @return null if no handler has been registered to match this encoding.
	 */
	@Nullable
	public TransferEncoding getTransferEncoding(@Nullable String encoding) {
		if (encoding == null || encoding.isEmpty()) {
			return IdentityTransferEncoding.getInstance();
		}

		return this.transferCodingHandlers.get(encoding.toLowerCase());
	}

	/**
	 * Registers a new HTTP content encoding handler.
	 *
	 * @param handler
	 *            the {@link ContentEncoding} to register.
	 */
	public void registerContentEncoding(ContentEncoding handler) {
		if (handler == null || handler.getId() == null || handler.getId().isEmpty()) {
			throw new IllegalArgumentException();
		}
		this.encodingHandlers.put(handler.getId().toLowerCase(), handler);
	}

	/**
	 * Registers a new HTTP transfer coding handler.
	 *
	 * @param handler
	 *            the {@link TransferEncoding} to register.
	 */
	public void registerTransferEncoding(TransferEncoding handler) {
		String handlerId = null;
		if (handler != null) {
			handlerId = handler.getId();
		}
		if (handler == null || handlerId == null || handlerId.isEmpty()) {
			throw new IllegalArgumentException();
		}
		this.transferCodingHandlers.put(handlerId.toLowerCase(), handler);
	}

	/**
	 * Returns the most suitable {@link ContentEncoding} to match the encodings described in
	 * <code>Accept-Encoding</code> header.
	 *
	 * @param encoding
	 *            is on the form <code>gzip, identity</code> or <code>gzip; q=0.8, identity; q=0.2</code>.
	 * @return the {@link ContentEncoding}, or <code>null</code> if no suitable handler can be found.
	 */
	@Nullable
	public ContentEncoding getAcceptEncodingHandler(@Nullable String encoding) {
		if (encoding == null || encoding.isEmpty()) {
			return null;
		}

		AcceptEncoding acceptEncoding = new AcceptEncoding();
		acceptEncoding.parse(encoding);

		// Try to return the most acceptable handler
		QualityArgument[] encodings = acceptEncoding.getEncodings();
		if (encodings == null) {
			return null;
		}
		int nbEncodings = encodings.length;
		boolean[] processed = new boolean[nbEncodings];
		for (int pass = nbEncodings - 1; pass >= 0; pass--) { // maximum number of passes
			float localMax = 0;
			int ptrMax = -1;
			for (int i = nbEncodings - 1; i >= 0; i--) {
				if (processed[i]) {
					continue;
				}
				QualityArgument arg = encodings[i];
				float qvalue = arg.getQuality();
				if (qvalue > localMax) {
					localMax = qvalue;
					ptrMax = i;
				}
			}
			processed[ptrMax] = true;

			// Try to get the handler
			ContentEncoding handler = getContentEncoding(encodings[ptrMax].getArgument());
			if (handler != null) {
				return handler;
			}
		}

		return null;
	}

}
