/*
 * Java
 *
 * Copyright 2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http.body;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;

import ej.annotation.Nullable;
import ej.hoka.http.support.Mime;

/**
 * A body parser for {@link Mime#MIME_MULTIPART_FORM_ENCODED_DATA} that parse parts as strings.
 */
public class MultipartStringsParser implements BodyParser<String[]> {

	/**
	 * Size of CR,LF,DASH,DASH RFC 1867 form-data sequence. See
	 * <a href="https://www.ietf.org/rfc/rfc1867.txt">RFC1867</a>.
	 */
	private static final int BOUNDARY_PREFIX = 4; // CR,LF,DASH,DASH

	/**
	 * Size of DASH,DASH RFC 1867 form-data sequence. See <a href="https://www.ietf.org/rfc/rfc1867.txt">RFC1867</a>.
	 */
	private static final int STREAM_TERMINATOR = 2; // DASH,DASH

	@Override
	public String[] parseBody(InputStream inputStream, @Nullable String contentType) throws IOException {
		if ((contentType != null) && contentType.startsWith(Mime.MIME_MULTIPART_FORM_ENCODED_DATA)) {
			String boundary = contentType.substring(contentType.indexOf(';') + 1);

			boundary = boundary
					.substring(boundary.indexOf(MultiPartBodyParser.BOUNDARY) + MultiPartBodyParser.BOUNDARY.length());
			String multipartBody = new StringBodyParser().parseBody(inputStream, null);
			return split(multipartBody, boundary);
		}

		throw new IllegalArgumentException();
	}

	private static String[] split(String toSplit, String separator) {
		int index = toSplit.indexOf(separator);
		List<String> parts = new ArrayList<>();

		while (index > -1) {
			int indexEnd = toSplit.indexOf(separator, index + separator.length());

			if (indexEnd != -1) {
				parts.add(
						toSplit.substring(index + separator.length() + STREAM_TERMINATOR, indexEnd - BOUNDARY_PREFIX));
			} else {
				parts.add(toSplit.substring(index + separator.length() + STREAM_TERMINATOR,
						toSplit.length() - STREAM_TERMINATOR));
			}
			index = indexEnd;
		}

		return parts.toArray(new String[parts.size()]);
	}

}
