/*
 * Java
 *
 * Copyright 2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bluetooth.callbacks;

import ej.bluetooth.BluetoothAdapter;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.callbacks.impl.DefaultServerCallbacks;

/**
 * The <code>ServerCallbacks</code> interface provides callbacks related to GATT server events.
 * <p>
 * After {@link BluetoothAdapter#addService adding a service} to the adapter, the application may
 * {@link BluetoothService#setServerCallbacks set the server callbacks} of the service in order to be notified of the
 * events related to this service.
 * <p>
 * This library provides a default implementation of this interface: {@link DefaultServerCallbacks}.
 */
public interface ServerCallbacks {

	/**
	 * Called when a read characteristic request is received.
	 *
	 * @param characteristic
	 *            the characteristic to read.
	 * @param device
	 *            the device which has sent the request.
	 * @see BluetoothCharacteristic#sendReadResponse(BluetoothDevice, int, byte[])
	 */
	void onReadRequest(BluetoothCharacteristic characteristic, BluetoothDevice device);

	/**
	 * Called when a write characteristic request is received.
	 *
	 * @param characteristic
	 *            the characteristic to write.
	 * @param device
	 *            the device which has sent the request.
	 * @param value
	 *            the value to write.
	 * @see BluetoothCharacteristic#sendWriteResponse(BluetoothDevice, int)
	 */
	void onWriteRequest(BluetoothCharacteristic characteristic, BluetoothDevice device, byte[] value);

	/**
	 * Called when a characteristic notification is sent.
	 *
	 * @param characteristic
	 *            the characteristic which has changed.
	 * @param device
	 *            the device to which the notification was sent.
	 * @param success
	 *            true if the notification was sent successfully, false otherwise.
	 */
	void onNotificationSent(BluetoothCharacteristic characteristic, BluetoothDevice device, boolean success);

	/**
	 * Called when a read descriptor request is received.
	 *
	 * @param descriptor
	 *            the descriptor to read.
	 * @param device
	 *            the device which has sent the request.
	 * @see BluetoothDescriptor#sendReadResponse(BluetoothDevice, int, byte[])
	 */
	void onReadRequest(BluetoothDescriptor descriptor, BluetoothDevice device);

	/**
	 * Called when a write descriptor request is received.
	 *
	 * @param descriptor
	 *            the descriptor to write.
	 * @param device
	 *            the device which has sent the request.
	 * @param value
	 *            the value to write.
	 * @see BluetoothDescriptor#sendWriteResponse(BluetoothDevice, int)
	 */
	void onWriteRequest(BluetoothDescriptor descriptor, BluetoothDevice device, byte[] value);
}
