/*
 * Java
 *
 * Copyright 2018 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.bluetooth.callbacks;

import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothDescriptor;
import ej.bluetooth.BluetoothDevice;
import ej.bluetooth.BluetoothService;
import ej.bluetooth.BluetoothStatus;
import ej.bluetooth.callbacks.impl.DefaultClientCallbacks;

/**
 * The <code>ClientCallbacks</code> interface provides callbacks related to GATT client events.
 * <p>
 * After {@link BluetoothDevice#discoverServices discovering the services} of a device, the application may
 * {@link BluetoothService#setClientCallbacks set the client callbacks} of a service in order to be notified of the
 * events related to this service.
 * <p>
 * This library provides a default implementation of this interface: {@link DefaultClientCallbacks}.
 */
public interface ClientCallbacks {

	/**
	 * Called when a response to a read characteristic request is received.
	 *
	 * @param characteristic
	 *            the characteristic which was read.
	 * @param status
	 *            the response status (see {@link BluetoothStatus}).
	 * @param value
	 *            the value of the characteristic.
	 * @see BluetoothCharacteristic#sendReadRequest()
	 */
	void onReadCompleted(BluetoothCharacteristic characteristic, int status, byte[] value);

	/**
	 * Called when a response to a write characteristic request is received.
	 *
	 * @param characteristic
	 *            the characteristic which was written.
	 * @param status
	 *            the response status (see {@link BluetoothStatus}).
	 * @see BluetoothCharacteristic#sendWriteRequest(byte[])
	 */
	void onWriteCompleted(BluetoothCharacteristic characteristic, int status);

	/**
	 * Called when a characteristic notification is received.
	 *
	 * @param characteristic
	 *            the characteristic which has changed.
	 * @param value
	 *            the value of the characteristic.
	 */
	void onNotificationReceived(BluetoothCharacteristic characteristic, byte[] value);

	/**
	 * Called when a response to a read descriptor request is received.
	 *
	 * @param descriptor
	 *            the descriptor which was read.
	 * @param status
	 *            the response status (see {@link BluetoothStatus}).
	 * @param value
	 *            the value of the descriptor.
	 * @see BluetoothDescriptor#sendReadRequest()
	 */
	void onReadCompleted(BluetoothDescriptor descriptor, int status, byte[] value);

	/**
	 * Called when a response to a write descriptor request is received.
	 *
	 * @param descriptor
	 *            the descriptor which was written.
	 * @param status
	 *            the response status (see {@link BluetoothStatus}).
	 * @see BluetoothDescriptor#sendWriteRequest(byte[])
	 */
	void onWriteCompleted(BluetoothDescriptor descriptor, int status);
}
