/*
 * Copyright (C) 2008 The Android Open Source Project
 * Copyright (C) 2015-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package android.net;

import ej.annotation.Nullable;
import ej.basictool.ArrayTools;

/**
 * Class that answers queries about the state of network connectivity. It also notifies applications when network
 * connectivity changes. Get an instance is dependent on the underlying system.
 */
public class ConnectivityManager {

	@Nullable
	private NetworkMapping defaultNetwork = null;
	private NetworkMapping[] networkMappings = new NetworkMapping[0];
	private RequestMapping[] requests = new RequestMapping[0];
	private NetworkCallback[] defaultCallbacks = new NetworkCallback[0];
	private final Object requestMutex = new Object();

	/**
	 */
	protected ConnectivityManager() {
	}

	/**
	 * Returns details about the currently active default data network. When connected, this network is the default
	 * route for outgoing connections. You should always check {@link NetworkInfo#isConnected()} before initiating
	 * network traffic. This may return {@code null} when there is no default network.
	 *
	 * @return a {@link NetworkInfo} object for the current default network or {@code null} if no network default
	 *         network is currently active
	 */
	@Nullable
	public NetworkInfo getActiveNetworkInfo() {
		NetworkMapping defaultNetwork = this.defaultNetwork;
		return (defaultNetwork == null) ? null : defaultNetwork.networkInfo;
	}

	/**
	 * Returns a {@link Network} object corresponding to the currently active default data network. In the event that
	 * the current active default data network disconnects, the returned {@link Network} object will no longer be
	 * usable. This will return null when there is no default network.
	 *
	 * @return a {@link Network} object for the current default network or <code>null</code> if no default network is
	 *         currently active
	 */
	@Nullable
	public Network getActiveNetwork() {
		NetworkMapping defaultNetwork = this.defaultNetwork;
		return (defaultNetwork == null) ? null : defaultNetwork.network;
	}

	/**
	 * Returns an array of all {@link Network} currently tracked by the framework.
	 *
	 * @return an array of {@link Network} objects.
	 */
	public Network[] getAllNetworks() {
		Network[] networks = new Network[0];
		for (NetworkMapping networkMapping : this.networkMappings) {
			networks = ArrayTools.add(networks, networkMapping.network);
		}
		return networks;
	}

	/**
	 * Get the {@link NetworkCapabilities} for the given Network. This will return <code>null</code> if the network is
	 * unknown.
	 *
	 * @param network
	 *            The {@link Network} object identifying the network in question.
	 * @return The {@link NetworkCapabilities} for the network, or <code>null</code>.
	 */
	@Nullable
	public NetworkCapabilities getNetworkCapabilities(Network network) {
		NetworkMapping mapping = getMapping(network);
		return (mapping != null) ? mapping.capabilities : null;
	}

	/**
	 * Returns connection status information about a particular Network.
	 *
	 * @param network
	 *            {@link Network} specifying which network in which you're interested.
	 * @return a {@link NetworkInfo} object for the requested network or <code>null</code> if the Network is not valid.
	 */
	@Nullable
	public NetworkInfo getNetworkInfo(Network network) {
		NetworkMapping mapping = getMapping(network);
		return (mapping != null) ? mapping.networkInfo : null;
	}

	/**
	 * Registers to receive notifications about changes in the system default network. The callbacks will continue to be
	 * called until either the application exits or {@link #unregisterNetworkCallback(NetworkCallback)} is called.
	 *
	 * @param networkCallback
	 *            The {@link NetworkCallback} that the system will call as the system default network changes. The
	 *            callback is invoked on the default internal Handler.
	 */
	public void registerDefaultNetworkCallback(NetworkCallback networkCallback) {
		synchronized (this.requestMutex) {
			if (ArrayTools.contains(this.defaultCallbacks, networkCallback)) {
				return;
			}
			this.defaultCallbacks = ArrayTools.add(this.defaultCallbacks, networkCallback);
		}
	}

	/**
	 * Registers to receive notifications about all networks which satisfy the given {@link NetworkRequest}. The
	 * callbacks will continue to be called until either the application exits or {@link #unregisterNetworkCallback} is
	 * called
	 *
	 * @param request
	 *            {@link NetworkRequest} describing this request.
	 * @param networkCallback
	 *            The {@link NetworkCallback} that the system will call as suitable networks change state.
	 */
	public void registerNetworkCallback(@Nullable NetworkRequest request, NetworkCallback networkCallback) {
		RequestMapping requestMapping = new RequestMapping(request, networkCallback);
		synchronized (this.requestMutex) {
			if (networkCallback != null && !ArrayTools.contains(this.requests, requestMapping)) {
				this.requests = ArrayTools.add(this.requests, requestMapping);
			}
		}
	}

	/**
	 * Unregisters callbacks about and possibly releases networks originating from {@link #registerNetworkCallback}
	 * calls. If the given {@code NetworkCallback} had previosuly been used with {@code #requestNetwork}, any networks
	 * that had been connected to only to satisfy that request will be disconnected.
	 *
	 * @param networkCallback
	 *            The {@link NetworkCallback} used when making the request.
	 */
	public void unregisterNetworkCallback(NetworkCallback networkCallback) {
		if (networkCallback == null) {
			throw new IllegalArgumentException();
		}
		synchronized (this.requestMutex) {
			this.defaultCallbacks = ArrayTools.remove(this.defaultCallbacks, networkCallback);
			RequestMapping[] strippedRequest = this.requests;
			for (RequestMapping requestMapping : this.requests) {
				if (requestMapping.networkCallback.equals(networkCallback)) {
					strippedRequest = ArrayTools.remove(strippedRequest, requestMapping);
				}
			}
			this.requests = strippedRequest;
		}
	}

	/**
	 * Base class for NetworkRequest callbacks. Used for notifications about network changes. Should be extended by
	 * applications wanting notifications.
	 */
	public static class NetworkCallback {
		/**
		 * Called when the framework connects and has declared new network ready for use. This callback may be called
		 * more than once if the {@link Network} that is satisfying the request changes.
		 *
		 * @param network
		 *            The {@link Network} of the satisfying network.
		 */
		public void onAvailable(Network network) {
			// Not used.
		}

		/**
		 * Called when the network the framework connected to for this request changes capabilities but still satisfies
		 * the stated need.
		 *
		 * @param network
		 *            The {@link Network} of the satisfying network.
		 * @param networkCapabilities
		 *            NetworkCapabilities: The new {@link NetworkCapabilities} for this network.
		 */
		public void onCapabilitiesChanged(Network network, NetworkCapabilities networkCapabilities) {
			// Not used.
		}

		/**
		 * Called when the framework has a hard loss of the network or when the graceful failure ends.
		 *
		 * @param network
		 *            The {@link Network} lost.
		 */
		public void onLost(Network network) {
			// Not used.
		}
	}

	/**
	 * Sets the active network.
	 *
	 * @param network
	 *            the network to set.
	 */
	protected void setActiveNetwork(Network network) {
		NetworkMapping mapping = getMapping(network);
		boolean isNew = mapping == null;
		if (isNew) {
			mapping = newMapping(network, false, new NetworkCapabilities());
		}
		assert mapping != null; // ensured just above
		NetworkMapping previousDefault = this.defaultNetwork;
		if (!mapping.equals(previousDefault)) {
			this.defaultNetwork = mapping;
			if (previousDefault != null) {
				Network previousNetwork = previousDefault.network;
				if (previousDefault.capabilities.hasCapability(NetworkCapabilities.NET_CAPABILITY_INTERNET)) {
					setCapabilities(previousDefault, new NetworkCapabilities());
					doNotifyCapabilitiesChange(previousNetwork, previousDefault.capabilities, this.defaultCallbacks);
				}
				if (previousDefault.networkInfo.isConnected()) {
					setAvailable(false, previousNetwork, previousDefault.networkInfo);
					doNotifyAvailabilityChange(previousNetwork, false, this.defaultCallbacks);
				}
			}
			if (!isNew) {
				doNotifyAvailabilityChange(network, mapping.networkInfo.isConnected(), this.defaultCallbacks);
				doNotifyCapabilitiesChange(mapping.network, mapping.capabilities, this.defaultCallbacks);
			}
		}
	}

	/**
	 * Notifies a network availability change.
	 *
	 * @param available
	 *            <code>true</code> if the network is available.
	 * @deprecated use {@link #notifyNetworkCallbacks(Network, boolean)}.
	 */
	@Deprecated
	protected void notifyNetworkCallbacks(boolean available) {
		notifyNetworkCallbacks(Network.NULL_NETWORK, available);
	}

	/**
	 * Notifies a network availability change.
	 *
	 * @param network
	 *            the network.
	 * @param available
	 *            <code>true</code> if the network is available.
	 */
	protected void notifyNetworkCallbacks(Network network, boolean available) {
		NetworkMapping mapping = getMapping(network);
		boolean isNew = mapping == null;
		if (isNew) {
			mapping = newMapping(network, available, new NetworkCapabilities());
		}
		assert mapping != null; // ensured just above
		setAvailable(available, mapping.network, mapping.networkInfo);
		if (mapping == this.defaultNetwork) {
			doNotifyAvailabilityChange(mapping.network, available, this.defaultCallbacks);
		}
		doNotifyAvailabilityChange(mapping.network, available,
				getNetworkCallBacks(mapping.capabilities, mapping.capabilities));
	}

	/**
	 * Notifies a network capability change.
	 *
	 * @param network
	 *            the network.
	 * @param capabilities
	 *            the network capabilities.
	 */
	protected void notifyNetworkCallbacks(Network network, NetworkCapabilities capabilities) {
		NetworkMapping mapping = getMapping(network);
		boolean isNew = mapping == null;
		if (isNew) {
			mapping = newMapping(network, false, capabilities);
		}
		assert mapping != null; // ensured just above
		NetworkCapabilities oldCapabilities = new NetworkCapabilities();
		oldCapabilities.setCapabilities(mapping.capabilities.getCapabilities());
		setCapabilities(mapping, capabilities);
		if (mapping == this.defaultNetwork) {
			doNotifyCapabilitiesChange(mapping.network, capabilities, this.defaultCallbacks);
		}
		doNotifyCapabilitiesChange(mapping.network, capabilities, getNetworkCallBacks(oldCapabilities, capabilities));
	}

	/**
	 * Sets the capabilities.
	 *
	 * @param network
	 *            the network.
	 * @param networkCapabilities
	 *            the registered capabilities.
	 * @param targetCapabilities
	 *            the capabilities to set.
	 */
	private void setCapabilities(NetworkMapping mapping, NetworkCapabilities targetCapabilities) {
		mapping.capabilities.setCapabilities(targetCapabilities.getCapabilities());
	}

	/**
	 * Sets the network availability.
	 *
	 * @param available
	 *            <code>true</code> if available.
	 * @param network
	 *            the network.
	 * @param networkInfo
	 *            the network info.
	 */
	protected void setAvailable(boolean available, @Nullable Network network, NetworkInfo networkInfo) {
		networkInfo.setConnected(available);
	}

	/**
	 * Notify the callback of a capabilities change.
	 *
	 * @param network
	 *            the network.
	 * @param capabilities
	 *            the capabilities.
	 * @param networkCallBacks
	 *            the callback to call.
	 */
	protected void doNotifyCapabilitiesChange(Network network, NetworkCapabilities capabilities,
			NetworkCallback[] networkCallBacks) {
		for (NetworkCallback networkCallback : networkCallBacks) {
			try {
				networkCallback.onCapabilitiesChanged(network, capabilities);
			} catch (Exception e) {
				// sanity
				e.printStackTrace();
			}
		}
	}

	/**
	 * Notify the networkCallbacks about an availability change.
	 *
	 * @param network
	 *            the network.
	 * @param available
	 *            the availability.
	 * @param networkCallbacks
	 *            the callback to call.
	 */
	protected void doNotifyAvailabilityChange(Network network, boolean available, NetworkCallback[] networkCallbacks) {
		if (available) {
			for (NetworkCallback networkCallback : networkCallbacks) {
				try {
					networkCallback.onAvailable(network);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		} else {
			for (NetworkCallback networkCallback : networkCallbacks) {
				try {
					networkCallback.onLost(network);
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
	}

	private NetworkMapping newMapping(Network network, boolean available, NetworkCapabilities capabilities) {
		NetworkMapping mapping;
		mapping = new NetworkMapping(network);
		mapping.networkInfo.setConnected(available);
		mapping.capabilities.setCapabilities(capabilities.getCapabilities());
		this.networkMappings = ArrayTools.add(this.networkMappings, mapping);
		return mapping;
	}

	private NetworkCallback[] getNetworkCallBacks(NetworkCapabilities oldCapabilities,
			NetworkCapabilities capabilities) {
		NetworkCallback[] callBacks = new NetworkCallback[0];
		for (RequestMapping requestMapping : this.requests) {
			if (requestMapping.fullfill(capabilities) || requestMapping.fullfill(oldCapabilities)) {
				callBacks = ArrayTools.add(callBacks, requestMapping.networkCallback);
			}
		}
		return callBacks;
	}

	private @Nullable NetworkMapping getMapping(Network network) {
		NetworkMapping mapping = null;
		if (network != null) {
			for (NetworkMapping networkMapping : this.networkMappings) {
				if (network.equals(networkMapping.network)) {
					mapping = networkMapping;
				}
			}
		}
		return mapping;
	}

	private static class NetworkMapping {
		private final Network network;
		private final NetworkInfo networkInfo = new NetworkInfo();
		private final NetworkCapabilities capabilities = new NetworkCapabilities();

		public NetworkMapping(Network network) {
			this.network = network;
		}
	}

	private static class RequestMapping {
		private final @Nullable NetworkRequest request;
		private final NetworkCallback networkCallback;

		public RequestMapping(@Nullable NetworkRequest request, NetworkCallback networkCallbacks) {
			this.request = request;
			this.networkCallback = networkCallbacks;
		}

		public boolean fullfill(NetworkCapabilities capabilities) {
			boolean fullfill = true;
			if (this.request != null) {
				for (int capability : this.request.networkCapabilities.getCapabilities()) {
					if (!capabilities.hasCapability(capability)) {
						fullfill = false;
						break;
					}
				}
			}
			return fullfill;
		}

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + this.networkCallback.hashCode();
			final NetworkRequest req = this.request;
			if (req != null) {
				result = prime * result + req.hashCode();
			}
			return result;
		}

		@Override
		public boolean equals(@Nullable Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			RequestMapping other = (RequestMapping) obj;
			if (!this.networkCallback.equals(other.networkCallback)) {
				return false;
			}
			if (this.request != null) {
				return this.request.equals(other.request);
			} else {
				return other.request == null;
			}
		}
	}
}
