/**
 * Java
 *
 * Copyright 2023 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package java.time;

/**
 * This class contains methods for performing basic numeric operations.
 */
class MathUtils {


	/**
	 * Returns the largest (closest to positive infinity) {@code long} value that is less than or equal to the algebraic
	 * quotient. There is one special case, if the dividend is the {@linkplain Long#MIN_VALUE Long.MIN_VALUE} and the
	 * divisor is {@code -1}, then integer overflow occurs and the result is equal to the {@code Long.MIN_VALUE}.
	 * <p>
	 * Normal integer division operates under the round to zero rounding mode (truncation). This operation instead acts
	 * under the round toward negative infinity (floor) rounding mode. The floor rounding mode gives different results
	 * than truncation when the exact result is negative.
	 * <p>
	 *
	 * @param x
	 *            the dividend
	 * @param y
	 *            the divisor
	 * @return the largest (closest to positive infinity) {@code long} value that is less than or equal to the algebraic
	 *         quotient.
	 * @throws ArithmeticException
	 *             if the divisor {@code y} is zero
	 * @see #floorMod(long, long)
	 * @see Math#floor(double)
	 * since 1.8
	 */
	static long floorDiv(long x, long y) {
		long r = x / y;
		// if the signs are different and modulo not zero, round down
		if ((x ^ y) < 0 && (r * y != x)) {
			r--;
		}
		return r;
	}


	/**
	 * Returns the floor modulus of the {@code long} arguments.
	 * <p>
	 * The floor modulus is {@code x - (floorDiv(x, y) * y)}, has the same sign as the divisor {@code y}, and is in the
	 * range of {@code -abs(y) < r < +abs(y)}.
	 *
	 * <p>
	 * The relationship between {@code floorDiv} and {@code floorMod} is such that:
	 * <ul>
	 * <li>{@code floorDiv(x, y) * y + floorMod(x, y) == x}
	 * </ul>
	 * <p>
	 *
	 * @param x
	 *            the dividend
	 * @param y
	 *            the divisor
	 * @return the floor modulus {@code x - (floorDiv(x, y) * y)}
	 * @throws ArithmeticException
	 *             if the divisor {@code y} is zero
	 * @see #floorDiv(long, long)
	 * since 1.8
	 */
	static long floorMod(long x, long y) {
		return x - floorDiv(x, y) * y;
	}

}
