/*
 * Copyright (c) 1997, 2014, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2021-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package java.security;

import java.security.spec.AlgorithmParameterSpec;

/**
 * The KeyPairGenerator class is used to generate pairs of public and private keys. Key pair generators are constructed
 * using the {@code getInstance} factory methods (static methods that return instances of a given class).
 *
 * <p>
 * A Key pair generator for a particular algorithm creates a public/private key pair that can be used with this
 * algorithm. It also associates algorithm-specific parameters with each of the generated keys.
 *
 * <p>
 * In case the client does not explicitly initialize the KeyPairGenerator (via a call to an {@code initialize} method),
 * each provider must supply (and document) a default initialization. For example, the <i>Sun</i> provider uses a
 * default modulus size (keysize) of 1024 bits.
 *
 * <p>
 * Note that this class is abstract and extends from {@code KeyPairGeneratorSpi} for historical reasons. Application
 * developers should only take notice of the methods defined in this {@code KeyPairGenerator} class; all the methods in
 * the superclass are intended for cryptographic service providers who wish to supply their own implementations of key
 * pair generators.
 *
 *
 * @author Benjamin Renaud
 *
 * @see java.security.spec.AlgorithmParameterSpec
 */

public abstract class KeyPairGenerator extends KeyPairGeneratorSpi {

	/**
	 * Creates a KeyPairGenerator object for the specified algorithm.
	 *
	 * @param algorithm
	 *            the standard string name of the algorithm. See the KeyPairGenerator section in the <a href=
	 *            "https://docs.oracle.com/en/java/javase/19/docs/specs/security/standard-names.html#keypairgenerator-algorithms">
	 *            Java Security Standard Algorithm Names Specification</a> for information about standard algorithm
	 *            names. Consult the release documentation for your implementation to see which algorithms are
	 *            supported.
	 */
	protected KeyPairGenerator(String algorithm) {
		throw new RuntimeException();
	}

	/**
	 * Returns a KeyPairGenerator object that generates public/private key pairs for the specified algorithm.
	 *
	 * <p>
	 * This method traverses the list of registered security Providers, starting with the most preferred Provider. A new
	 * KeyPairGenerator object encapsulating the KeyPairGeneratorSpi implementation from the first Provider that
	 * supports the specified algorithm is returned.
	 *
	 * <p>
	 * Note that the list of registered providers may be retrieved via the { Security#getProviders()
	 * Security.getProviders()} method.
	 *
	 * @param algorithm
	 *            the standard string name of the algorithm. See the KeyPairGenerator section in the <a href=
	 *            "https://docs.oracle.com/en/java/javase/19/docs/specs/security/standard-names.html#keypairgenerator-algorithms">
	 *            Java Security Standard Algorithm Names Specification</a> for information about standard algorithm
	 *            names. Consult the release documentation for your implementation to see which algorithms are
	 *            supported.
	 *
	 * @return the new KeyPairGenerator object.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if no Provider supports a KeyPairGeneratorSpi implementation for the specified algorithm.
	 */
	public static KeyPairGenerator getInstance(String algorithm) throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	void disableFailover() {
		throw new RuntimeException();
	}

	/**
	 * Initializes the key pair generator for a certain keysize using a default parameter set and the
	 * {@code SecureRandom} implementation of the highest-priority installed provider as the source of randomness. (If
	 * none of the installed providers supply an implementation of {@code SecureRandom}, a system-provided source of
	 * randomness is used.)
	 *
	 * @param keysize
	 *            the keysize. This is an algorithm-specific metric, such as modulus length, specified in number of
	 *            bits.
	 *
	 * @exception InvalidParameterException
	 *                if the {@code keysize} is not supported by this KeyPairGenerator object.
	 */
	public void initialize(int keysize) {
		throw new RuntimeException();
	}

	/**
	 * Initializes the key pair generator for a certain keysize with the given source of randomness (and a default
	 * parameter set).
	 *
	 * @param keysize
	 *            the keysize. This is an algorithm-specific metric, such as modulus length, specified in number of
	 *            bits.
	 * @param random
	 *            the source of randomness.
	 *
	 * @exception InvalidParameterException
	 *                if the {@code keysize} is not supported by this KeyPairGenerator object.
	 *
	 * @since 1.2
	 */
	@Override
	public void initialize(int keysize, SecureRandom random) {
		throw new RuntimeException();
	}

	/**
	 * Initializes the key pair generator using the specified parameter set and the {@code SecureRandom} implementation
	 * of the highest-priority installed provider as the source of randomness. (If none of the installed providers
	 * supply an implementation of {@code SecureRandom}, a system-provided source of randomness is used.).
	 *
	 * <p>
	 * This concrete method has been added to this previously-defined abstract class. This method calls the
	 * KeyPairGeneratorSpi
	 * {@link KeyPairGeneratorSpi#initialize( java.security.spec.AlgorithmParameterSpec, java.security.SecureRandom)
	 * initialize} method, passing it {@code params} and a source of randomness (obtained from the highest-priority
	 * installed provider or system-provided if none of the installed providers supply one). That {@code initialize}
	 * method always throws an UnsupportedOperationException if it is not overridden by the provider.
	 *
	 * @param params
	 *            the parameter set used to generate the keys.
	 *
	 * @exception InvalidAlgorithmParameterException
	 *                if the given parameters are inappropriate for this key pair generator.
	 *
	 * @since 1.2
	 */
	public void initialize(AlgorithmParameterSpec params) throws InvalidAlgorithmParameterException {
		throw new RuntimeException();
	}

	/**
	 * Initializes the key pair generator with the given parameter set and source of randomness.
	 *
	 * <p>
	 * This concrete method has been added to this previously-defined abstract class. This method calls the
	 * KeyPairGeneratorSpi
	 * {@link KeyPairGeneratorSpi#initialize( java.security.spec.AlgorithmParameterSpec, java.security.SecureRandom)
	 * initialize} method, passing it {@code params} and {@code random}. That {@code initialize} method always throws an
	 * UnsupportedOperationException if it is not overridden by the provider.
	 *
	 * @param params
	 *            the parameter set used to generate the keys.
	 * @param random
	 *            the source of randomness.
	 *
	 * @exception InvalidAlgorithmParameterException
	 *                if the given parameters are inappropriate for this key pair generator.
	 *
	 * @since 1.2
	 */
	@Override
	public void initialize(AlgorithmParameterSpec params, SecureRandom random)
			throws InvalidAlgorithmParameterException {
		throw new RuntimeException();
	}

	/**
	 * Generates a key pair.
	 *
	 * <p>
	 * If this KeyPairGenerator has not been initialized explicitly, provider-specific defaults will be used for the
	 * size and other (algorithm-specific) values of the generated keys.
	 *
	 * <p>
	 * This will generate a new key pair every time it is called.
	 *
	 * <p>
	 * This method is functionally equivalent to {@link #generateKeyPair() generateKeyPair}.
	 *
	 * @return the generated key pair
	 *
	 * @since 1.2
	 */
	public final KeyPair genKeyPair() {
		throw new RuntimeException();
	}

	/**
	 * Generates a key pair.
	 *
	 * <p>
	 * If this KeyPairGenerator has not been initialized explicitly, provider-specific defaults will be used for the
	 * size and other (algorithm-specific) values of the generated keys.
	 *
	 * <p>
	 * This will generate a new key pair every time it is called.
	 *
	 * <p>
	 * This method is functionally equivalent to {@link #genKeyPair() genKeyPair}.
	 *
	 * @return the generated key pair
	 */
	@Override
	public KeyPair generateKeyPair() {
		// This does nothing (except returning null), because either:
		//
		// 1. the implementation object returned by getInstance() is an
		// instance of KeyPairGenerator which has its own implementation
		// of generateKeyPair (overriding this one), so the application
		// would be calling that method directly, or
		//
		// 2. the implementation returned by getInstance() is an instance
		// of Delegate, in which case generateKeyPair is
		// overridden to invoke the corresponding SPI method.
		//
		// (This is a special case, because in JDK 1.1.x the generateKeyPair
		// method was used both as an API and a SPI method.)
		throw new RuntimeException();
	}

}
