/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.ThirdEventThread;
import ej.fp.util.ThirdEventThread.ThirdEventWidget;
import ej.microui.event.EventButton;

/**
 * Extended button widget which manages <i>repeat</i> button press events.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link RepeatButtonListener} to send the button
 * events,</li>
 * <li>The widget label must be an integer. It allows to create a MicroUI event using this button ID.</li>
 * </ul>
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label"), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "skin"), @WidgetAttribute(name = "pushedSkin"),
		@WidgetAttribute(name = "repeatPeriod", isOptional = true),
		@WidgetAttribute(name = "filter", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class RepeatButton extends Button implements ThirdEventWidget {

	/**
	 * Interface that handle repeat button press events.
	 */
	public static interface RepeatButtonListener extends ButtonListener {

		/**
		 * The specified button has been pressed again.
		 *
		 * @param widget
		 *            the button that has been pressed again.
		 */
		void repeat(RepeatButton widget);
	}

	/**
	 * Default implementation of {@link RepeatButtonListener}.
	 *
	 * @see ej.fp.widget.Button.ButtonListenerToButtonEvents
	 */
	public static class RepeatButtonListenerToButtonEvents extends ButtonListenerToButtonEvents
			implements RepeatButtonListener {

		@Override
		public void repeat(RepeatButton widget) {
			EventButton.sendRepeatedEvent(getMicroUIGeneratorTag(), widget.getID());
		}
	}

	private final ThirdEventThread repeatThread;
	private int repeatPeriod;

	/**
	 * Creates a repeat button widget.
	 * <p>
	 * The default repeat period is 200ms.
	 */
	public RepeatButton() {
		this.repeatThread = new ThirdEventThread(this, true);
		this.repeatPeriod = 200;
	}

	/**
	 * Sets the repeat event period.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param period
	 *            a time in milliseconds.
	 */
	public void setRepeatPeriod(int period) {
		if (this.repeatPeriod <= 0) {
			throw new IllegalArgumentException("Repeat period cannot be negative.");
		}
		this.repeatPeriod = period;
	}

	@Override
	public void setListenerClass(String listenerClassName) {
		setListenerClass(RepeatButtonListener.class, listenerClassName);
	}

	@Override
	public void start() {
		super.start();
		this.repeatThread.start();
	}

	@Override
	public void dispose() {
		this.repeatThread.dispose();
		super.dispose();
	}

	@Override
	protected Object newDefaultListener() {
		return new RepeatButtonListenerToButtonEvents();
	}

	@Override
	public void sendThirdEvent() {
		((RepeatButtonListener) getListener()).repeat(this);
	}

	@Override
	public int getSleepPeriod() {
		return this.repeatPeriod;
	}

	@Override
	protected void press() {
		ThirdEventThread repeatThread = this.repeatThread;
		if (!repeatThread.isSendingThirdEvent()) {
			super.press();
			repeatThread.wakeup();
		}
		// else don't send again a press event if we are sending repeat events
	}

	@Override
	protected void release() {
		this.repeatThread.goToSleep();
		super.release();
	}
}
