/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.fp.util;

import ej.fp.Device;
import ej.fp.FrontPanel;
import ej.fp.Widget;

/**
 * This extension of widget holds several methods to instanciate a listener to the widget.
 * <p>
 * This listener is set in the front panel file (<code>*.fp</code>) using the method
 * {@link #setListenerClass(Class, String)}. This user listener (front panel project user class) is checked (exists,
 * implements right interface or extends right class, can be instanciated etc.) before using it.
 * <p>
 * The listener must be instanciated in {@link #start()} method calling the method {@link #newListener(Class)}. It
 * should not be instanciated earlier because user code may call {@link Device#getDevice()}: this method throws an
 * exception until the device is fully created.
 * <p>
 * Setting an user class (using {@link #setListenerClass(Class, String)} is optional. When not set, this class asks to
 * subclass to create a default listener. This default listener cannot be null. At least, it should describes the widget
 * event printing something in the standard output stream ({@link FrontPanel#out}).
 */
@SuppressWarnings("nls")
public abstract class WidgetWithListener extends Widget {

	// name (package + class) of listener
	private String listenerClassName; // null init

	/**
	 * Sets the listener class name (package + class). This class must exist otherwise an
	 * {@link IllegalArgumentException} is thrown.
	 *
	 * @param expectedType
	 *            the class the listener has to implement or extends.
	 * @param listenerClassName
	 *            the class name to load.
	 */
	public void setListenerClass(Class<?> expectedType, String listenerClassName) {
		FrontPanel.getFrontPanel().verifyUserClass(getClass(), expectedType, listenerClassName);
		this.listenerClassName = listenerClassName;
	}

	/**
	 * Creates a new listener which has to implement / extend the given type.
	 * <p>
	 * First, the implementation tries to create the listener whose class name has been set by
	 * {@link #setListenerClass(Class, String)}. On class loading error, an exception is thrown: the user class must be
	 * valid.
	 * <p>
	 * If the user class name has not been set, the implementation asks the subclass to create a default listener. This
	 * listener must not be null and has to implement / extend given class.
	 *
	 * @param expectedType
	 *            the class the listener has to implement / extend.
	 * @return the created listener.
	 */
	protected synchronized <T> T newListener(Class<T> expectedType) {
		T listener;
		if (this.listenerClassName == null) {
			listener = createDefaultListener(expectedType);
		} else {
			listener = createUserListener(expectedType);
		}
		return listener;
	}

	/**
	 * Lets subclass creates a default listener. This method must not return null. The listener should at least prints
	 * some traces in the standard output stream.
	 *
	 * @return a default listener.
	 */
	protected Object newDefaultListener() {
		return null;
	}

	private <T> T createUserListener(Class<T> expectedType) {
		return FrontPanel.getFrontPanel().newUserInstance(getClass(), expectedType, this.listenerClassName);
	}

	private <T> T createDefaultListener(Class<T> expectedType) {

		T listener = null;

		try {

			// let subclass creates the default listener
			Object o = newDefaultListener();

			if (o == null) {
				// fatal error
				throw new IllegalStateException(
						"A listener which implements/extends " + expectedType.getName() + " is required.");
			}

			listener = expectedType.cast(o);

		} catch (ClassCastException e) {
			throw new IllegalArgumentException(this.listenerClassName + " cannot be cast to " + expectedType.getName());
		}

		return listener;
	}
}
