/*
 * Java
 *
 * Copyright 2021-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.BufferedImage;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;

/**
 * Tests all blend modes with multiple alpha values.
 */
public class TestBlendMode {

	private static final int BACKGROUND_COLOR = 0xFF8000; // orange
	private static final int FOREGROUND_COLOR = 0x306090; // blue
	private static final int[] ALPHAS = new int[] { 0, 63, 127, 191, 255 };

	// expected colors (computed on frontpanel with RGB888 display)
	private static final int[] SRC_EXPECTED_COLORS = //
			new int[] { 0x000000, 0x0b1723, 0x172f47, 0x23476b, 0x306090 };
	private static final int[] SRC_OVER_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xcb7723, 0x976f47, 0x63676b, 0x306090 };
	private static final int[] DST_OVER_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xff8000, 0xff8000, 0xff8000, 0xff8000 };
	private static final int[] SRC_IN_EXPECTED_COLORS = //
			new int[] { 0x000000, 0x0b1723, 0x172f47, 0x23476b, 0x306090 };
	private static final int[] DST_IN_EXPECTED_COLORS = //
			new int[] { 0x000000, 0x3f1f00, 0x7f3f00, 0xbf5f00, 0xff8000 };
	private static final int[] DST_OUT_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xc06000, 0x804000, 0x402000, 0x000000 };
	private static final int[] PLUS_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xff9723, 0xffaf47, 0xffc76b, 0xffe090 };
	private static final int[] SCREEN_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xff8c23, 0xff9847, 0xffa46b, 0xffb090 };
	private static final int[] MULTIPLY_EXPECTED_COLORS = //
			new int[] { 0xff8000, 0xcb6b00, 0x975700, 0x634300, 0x303000 };

	/**
	 * Starts MicroUI and resets the content of the screen to black.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	@Before
	public static void preTest() {
		final GraphicsContext gc = Display.getDisplay().getGraphicsContext();
		gc.resetTranslation();
		TestUtilities.clearScreen(BACKGROUND_COLOR);
	}

	@After
	public static void postTest() {
		Display.getDisplay().flush();
	}

	/**
	 * Tests with {@link BlendMode#SRC} blend mode (target is the display).
	 */
	@Test
	public static void testSrcBlendMode() {
		testBlendModeOnDisplay(BlendMode.SRC, SRC_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SRC_OVER} blend mode (target is the display).
	 */
	@Test
	public static void testSrcOverBlendMode() {
		testBlendModeOnDisplay(BlendMode.SRC_OVER, SRC_OVER_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_OVER} blend mode (target is the display).
	 */
	@Test
	public static void testDstOverBlendMode() {
		testBlendModeOnDisplay(BlendMode.DST_OVER, DST_OVER_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SRC_IN} blend mode (target is the display).
	 */
	@Test
	public static void testSrcInBlendMode() {
		testBlendModeOnDisplay(BlendMode.SRC_IN, SRC_IN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_IN} blend mode (target is the display).
	 */
	@Test
	public static void testDstInBlendMode() {
		testBlendModeOnDisplay(BlendMode.DST_IN, DST_IN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_OUT} blend mode (target is the display).
	 */
	@Test
	public static void testDstOutBlendMode() {
		testBlendModeOnDisplay(BlendMode.DST_OUT, DST_OUT_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#PLUS} blend mode (target is the display).
	 */
	@Test
	public static void testPlusBlendMode() {
		testBlendModeOnDisplay(BlendMode.PLUS, PLUS_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SCREEN} blend mode (target is the display).
	 */
	@Test
	public static void testScreenBlendMode() {
		testBlendModeOnDisplay(BlendMode.SCREEN, SCREEN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#MULTIPLY} blend mode (target is the display).
	 */
	@Test
	public static void testMultiplyBlendMode() {
		testBlendModeOnDisplay(BlendMode.MULTIPLY, MULTIPLY_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SRC} blend mode (target is a buffered image).
	 */
	@Test
	public static void testSrcBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.SRC, SRC_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SRC_OVER} blend mode (target is a buffered image).
	 */
	@Test
	public static void testSrcOverBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.SRC_OVER, SRC_OVER_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_OVER} blend mode (target is a buffered image).
	 */
	@Test
	public static void testDstOverBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.DST_OVER, DST_OVER_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SRC_IN} blend mode (target is a buffered image).
	 */
	@Test
	public static void testSrcInBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.SRC_IN, SRC_IN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_IN} blend mode (target is a buffered image).
	 */
	@Test
	public static void testDstInBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.DST_IN, DST_IN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#DST_OUT} blend mode (target is a buffered image).
	 */
	@Test
	public static void testDstOutBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.DST_OUT, DST_OUT_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#PLUS} blend mode (target is a buffered image).
	 */
	@Test
	public static void testPlusBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.PLUS, PLUS_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#SCREEN} blend mode (target is a buffered image).
	 */
	@Test
	public static void testScreenBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.SCREEN, SCREEN_EXPECTED_COLORS);
	}

	/**
	 * Tests with {@link BlendMode#MULTIPLY} blend mode (target is a buffered image).
	 */
	@Test
	public static void testMultiplyBlendModeOnBufferedImage() {
		testBlendModeOnBufferedImage(BlendMode.MULTIPLY, MULTIPLY_EXPECTED_COLORS);
	}

	private static void testBlendModeOnDisplay(BlendMode blendMode, int[] expectedColors) {
		// fill path with multiple alpha values
		Display display = Display.getDisplay();
		GraphicsContext gc = display.getGraphicsContext();
		gc.setTranslation(70 + blendMode.ordinal() * 30, 100);
		Path path = createSquarePath(30);
		for (int i = 0; i < ALPHAS.length; i++) {
			gc.setColor(BACKGROUND_COLOR);
			Painter.fillRectangle(gc, 0, 0, 30, 30);
			gc.setColor(FOREGROUND_COLOR);
			VectorGraphicsPainter.fillPath(gc, path, new Matrix(), FillType.WINDING, ALPHAS[i], blendMode);
			gc.translate(0, 30);
		}

		// flush for debug
		display.flush();

		// check the color for each alpha value
		gc.setTranslation(70 + blendMode.ordinal() * 30, 100);
		checkAllOpacities(gc, blendMode, expectedColors);
	}

	private static void testBlendModeOnBufferedImage(BlendMode blendMode, int[] expectedColors) {

		Display display = Display.getDisplay();
		GraphicsContext gc = display.getGraphicsContext();
		gc.setTranslation(70 + blendMode.ordinal() * 30, 100);

		try (BufferedImage image = new BufferedImage(30, 30)) {

			Path path = createSquarePath(image.getWidth());
			GraphicsContext igc = image.getGraphicsContext();

			for (int i = 0; i < ALPHAS.length; i++) {

				// fill path with alpha value
				igc.setColor(BACKGROUND_COLOR);
				Painter.fillRectangle(igc, 0, 0, 30, 30);
				igc.setColor(FOREGROUND_COLOR);
				VectorGraphicsPainter.fillPath(igc, path, new Matrix(), FillType.WINDING, ALPHAS[i], blendMode);

				// check the color in the buffered image
				checkOneOpacity(igc, blendMode, expectedColors, i);

				// draw the buffered image in the display (see next check after the loop)
				Painter.drawImage(gc, image, 0, 0);
				gc.translate(0, 30);
			}
		}

		// flush for debug
		display.flush();

		// check the color for each alpha value
		gc.setTranslation(70 + blendMode.ordinal() * 30, 100);
		checkAllOpacities(gc, blendMode, expectedColors);
	}

	private static void checkOneOpacity(GraphicsContext gc, BlendMode blendMode, int[] expectedColors, int index) {

		// check if output opacity is fully opaque
		int pix = gc.readPixel(20, 20);
		assertEquals(blendMode + " result fully opaque", TestUtilities.getColorWithAlpha(pix, GraphicsContext.OPAQUE),
				pix);

		// check the R-G-B components
		TestUtilities.check(blendMode + " alpha=" + ALPHAS[index], 20, 20, gc, expectedColors[index]);
	}

	private static void checkAllOpacities(GraphicsContext gc, BlendMode blendMode, int[] expectedColors) {
		for (int i = 0; i < ALPHAS.length; i++) {
			checkOneOpacity(gc, blendMode, expectedColors, i);
			gc.translate(0, 30);
		}
	}

	private static final Path createSquarePath(int size) {
		Path path = new Path();
		path.lineTo(0, size);
		path.lineTo(size, size);
		path.lineTo(size, 0);
		path.close();
		return path;
	}
}
