/*
 * Java
 *
 * Copyright 2019-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.event;

import ej.fp.FrontPanel;

/**
 * This class holds all methods to send MicroUI Button events to MicroUI application.
 */
@SuppressWarnings("nls")
public class EventButton {

	/**
	 * Usual MicroUI Buttons generator tag used in platform configuration project (<code>microui.xml</code>).
	 */
	public static final String COMMON_MICROUI_GENERATOR_TAG = "BUTTONS";

	// MicroUI button events representation
	private static final int BUTTONS_DRIVER_ACTION_MASK = 0x0000ff00;
	private static final int BUTTONS_DRIVER_ACTION_SHIFT = 8;
	private static final int BUTTONS_DRIVER_BUTTON_MASK = 0x000000ff;
	private static final int BUTTONS_DRIVER_BUTTON_SHIFT = 0;
	private static final int BUTTONS_DRIVER_PRESS = 0x00;
	private static final int BUTTONS_DRIVER_RELEASE = 0x01;
	private static final int BUTTONS_DRIVER_LONG = 0x02;
	private static final int BUTTONS_DRIVER_REPEAT = 0x03;

	private EventButton() {
		// hide implicit constructor
	}

	/**
	 * Send a PRESSED event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 * @see #sendPressedEvent(String, int)
	 */
	public static void sendPressedEvent(int buttonId) {
		sendPressedEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId);
	}

	/**
	 * Send a PRESSED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Buttons</code> event
	 * generator.
	 *
	 * @param buttonsEventGenerator
	 *            the MicroUI <code>Buttons</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 */
	public static void sendPressedEvent(String buttonsEventGenerator, int buttonId) {
		try {
			sendEvent(buttonsEventGenerator, buttonId, BUTTONS_DRIVER_PRESS);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(buttonsEventGenerator, buttonId, "press");
		}
	}

	/**
	 * Send a RELEASED event to the event generator with the MicroUI generator tag
	 * {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 * @see #sendReleasedEvent(String, int)
	 */
	public static void sendReleasedEvent(int buttonId) {
		sendReleasedEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId);
	}

	/**
	 * Send a RELEASED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Buttons</code> event
	 * generator.
	 *
	 * @param buttonsEventGenerator
	 *            the MicroUI <code>Buttons</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 */
	public static void sendReleasedEvent(String buttonsEventGenerator, int buttonId) {
		try {
			sendEvent(buttonsEventGenerator, buttonId, BUTTONS_DRIVER_RELEASE);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(buttonsEventGenerator, buttonId, "release");
		}
	}

	/**
	 * Send a REPEATED event to the event generator with the MicroUI generator tag
	 * {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 * @see #sendRepeatedEvent(String, int)
	 */
	public static void sendRepeatedEvent(int buttonId) {
		sendRepeatedEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId);
	}

	/**
	 * Send a REPEATED event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Buttons</code> event
	 * generator.
	 *
	 * @param buttonsEventGenerator
	 *            the MicroUI <code>Buttons</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 */
	public static void sendRepeatedEvent(String buttonsEventGenerator, int buttonId) {
		try {
			sendEvent(buttonsEventGenerator, buttonId, BUTTONS_DRIVER_REPEAT);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(buttonsEventGenerator, buttonId, "repeat");
		}
	}

	/**
	 * Send a LONG event to the event generator with the MicroUI generator tag {@link #COMMON_MICROUI_GENERATOR_TAG}.
	 *
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 * @see #sendLongEvent(String, int)
	 */
	public static void sendLongEvent(int buttonId) {
		sendLongEvent(COMMON_MICROUI_GENERATOR_TAG, buttonId);
	}

	/**
	 * Send a LONG event to the event generator.
	 * <p>
	 * The result of this method is unknown when the given event generator is not a MicroUI <code>Buttons</code> event
	 * generator.
	 *
	 * @param buttonsEventGenerator
	 *            the MicroUI <code>Buttons</code> event generator.
	 * @param buttonId
	 *            the button ID, between 0 and 255.
	 *
	 */
	public static void sendLongEvent(String buttonsEventGenerator, int buttonId) {
		try {
			sendEvent(buttonsEventGenerator, buttonId, BUTTONS_DRIVER_LONG);
		} catch (IllegalArgumentException e) {
			// invalid MicroUI events generator ID
			// -> print a trace
			traceEvent(buttonsEventGenerator, buttonId, "long press");
		}
	}

	private static void sendEvent(String buttonsEventGenerator, int buttonId, int event) {
		LLUIInput.sendEvent(buttonsEventGenerator,
				((event << BUTTONS_DRIVER_ACTION_SHIFT) & BUTTONS_DRIVER_ACTION_MASK)
						| ((buttonId << BUTTONS_DRIVER_BUTTON_SHIFT) & BUTTONS_DRIVER_BUTTON_MASK));
	}

	private static void traceEvent(String buttonsEventGenerator, int buttonId, String desc) {
		StringBuilder buf = new StringBuilder();
		buf.append("[");
		buf.append(EventButton.class.getName());
		buf.append("] ");
		buf.append(desc);
		buf.append(" button ");
		buf.append(buttonId);
		buf.append(" (");
		buf.append(buttonsEventGenerator);
		buf.append(")");
		FrontPanel.out.println(buf.toString());
	}
}
