/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;

/**
 * This is the super class of the front panel engines. According where the front panel framework is running (Front Panel
 * plugin or MicroEJ application), the front panel engine is different. This class allows to make an abstraction between
 * all implementations.
 * <p>
 * The implementation has the responsability to create the unique device according the front panel file
 * (<code>*.fp</code>) written by the user.
 * <p>
 * The steps to create a device are:
 * <ol>
 * <li>creates the front panel instance,</li>
 * <li>parses the front panel file (<code>*.fp</code>),</li>
 * <li>creates all widgets one by one and finalizes them,</li>
 * <li>starts the simulation starting all widgets.</li>
 * </ol>
 */
@SuppressWarnings("nls")
public abstract class FrontPanel {

	/**
	 * This static field makes an abstraction on the standard output stream where print some traces. It should be used
	 * instead of {@link System#out}.
	 */
	public static final PrintStream out;

	// unique instance of front panel engine
	private static final FrontPanel FrontPanel;

	/**
	 * Clinit block to create the unique front panel engine.
	 */
	static {

		try {
			// the engine class must be called "ej.fp.FrontPanelImpl"
			FrontPanel = (FrontPanel) Class.forName("ej.fp.FrontPanelImpl").newInstance();
		} catch (InstantiationException | IllegalAccessException | ClassNotFoundException e) {
			// throw a runtime error because simulation cannot start
			throw new IllegalStateException("Cannot create instance of FrontPanel.");
		}

		// retrieve the traces output stream
		out = FrontPanel.getPrintStream();

		// start the simulation
		FrontPanel.start();
	}

	/**
	 * Gets the unique instance of the front panel engine.
	 *
	 * @return the front panel engine.
	 */
	public static FrontPanel getFrontPanel() {
		return FrontPanel;
	}

	/**
	 * Gets the unique device created from the front panel file (<code>*.fp</code>).
	 *
	 * @return the unique device instance.
	 * @throws IllegalStateException
	 *             thrown when the caller tries to get the device whereas the device is not fully created yet. This
	 *             method can not be called before the call to {@link FrontPanel#start()} performed by the clinit (that
	 *             means <i>after</i> the device and its widgets creation and finalization.
	 */
	public abstract Device getDevice() throws IllegalStateException;

	/**
	 * Starts the simulation. Should only be called by the {@link FrontPanel} clinit.
	 */
	protected abstract void start();

	/**
	 * Gets the output streams where print the traces.
	 *
	 * @return the traces output stream.
	 */
	protected abstract PrintStream getPrintStream();

	/**
	 * Asks to repaint all device area.
	 */
	public abstract void repaintDevice();

	/**
	 * Asks to repaint a part of device area.
	 *
	 * @param x
	 *            x coordinate of area to repaint.
	 * @param y
	 *            y coordinate of area to repaint.
	 * @param width
	 *            width of area to repaint.
	 * @param height
	 *            height of area to repaint.
	 */
	public abstract void repaintDevice(int x, int y, int width, int height);

	/**
	 * Asks to repaint all widget area.
	 *
	 * @param w
	 *            the widget to repaint.
	 */
	public abstract void repaintWidget(Widget w);

	/**
	 * Asks to repaint a part of widget area.
	 *
	 * @param w
	 *            the widget to repaint.
	 * @param x
	 *            x coordinate of area to repaint.
	 * @param y
	 *            y coordinate of area to repaint.
	 * @param width
	 *            width of area to repaint.
	 * @param height
	 *            height of area to repaint.
	 */
	public abstract void repaintWidget(Widget w, int x, int y, int width, int height);

	/**
	 * Loads the image defined by the <code>stream</code> in a platform-specific {@link Image}.
	 *
	 * @param stream
	 *            the input stream of the image to load.
	 * @return the device-specific {@link Image} created.
	 * @throws IOException
	 *             if an error occur during the loading of the image.
	 */
	public abstract Image newImage(InputStream stream) throws IOException;

	/**
	 * Creates a new platform-specific {@link Image} with the specified size and initial color.
	 *
	 * @param width
	 *            the width of the wanted image.
	 * @param height
	 *            the height of the wanted image.
	 * @param color
	 *            the initial color of the wanted image.
	 * @param isTransparent
	 *            specify if the image is transparent.
	 * @return the device-specific {@link Image} created.
	 */
	public abstract Image newImage(int width, int height, int color, boolean isTransparent);

	/**
	 * Creates a copy of given image in a new platform-specific {@link Image}. The background color is black (0x0) and
	 * the image is transparent.
	 *
	 * @param image
	 *            the image to copy.
	 * @return the device-specific {@link Image} created.
	 */
	public Image newImage(Image image) {
		Image newImage = newImage(image.getWidth(), image.getHeight(), 0, true);
		newImage.drawImage(image);
		return newImage;
	}

	/**
	 * Checks if the class whose name is classname is a subclass of expected type and if this class is available in the
	 * same class loader than caller.
	 *
	 * @param caller
	 *            the caller class, used to retrieve the caller class loader.
	 * @param expectedType
	 *            the type the class to check has to extend or implement.
	 * @param classname
	 *            the class name to check.
	 * @throws IllegalArgumentException
	 *             if at least one condition is not respected.
	 */
	public void verifyUserClass(Class<?> caller, Class<?> expectedType, String classname)
			throws IllegalArgumentException {
		// getUserClass already performs the checks
		getUserClass(caller, expectedType, classname);
	}

	/**
	 * Gets the class whose name is classname. The class must be a subclass of expected type and the class must be
	 * available in the same class loader than caller.
	 *
	 * @param caller
	 *            the caller class, used to retrieve the caller class loader.
	 * @param expectedType
	 *            the type the class to check has to extend or implement.
	 * @param classname
	 *            the class name to check.
	 * @return the expected class.
	 * @throws IllegalArgumentException
	 *             if at least one condition is not respected.
	 */
	public abstract Class<?> getUserClass(Class<?> caller, Class<?> expectedType, String classname)
			throws IllegalArgumentException;

	/**
	 * Instanciates the class whose name is classname. The class must be a subclass of expected type and the class must
	 * be available in the same class loader than caller.
	 *
	 * @param caller
	 *            the caller class, used to retrieve the caller class loader.
	 * @param expectedType
	 *            the type the class to check has to extend or implement.
	 * @param classname
	 *            the class name to check.
	 * @return the expected instance.
	 * @throws IllegalArgumentException
	 *             if at least one condition is not respected.
	 */
	public abstract <T> T newUserInstance(Class<?> caller, Class<T> expectedType, String classname)
			throws IllegalArgumentException;

	/**
	 * Checks if given version is compatible with the compatibilityVersion.
	 * <p>
	 * This method is used by front panel framework to ensure the engine in plugin (designer mode) or the engine in
	 * MicroEJ platform (application mode) is compatible with the framework fetched by the front panel user project
	 * (project which contains the front panel file (<code>*.fp</code>)).
	 * <p>
	 * The versions are compatible when:
	 * <ul>
	 * <li>they have the same <i>major</i> value,</li>
	 * <li>the <i>minor</i> value of version is smaller or equals than the <i>minor</i> value of
	 * compatibilityVersion,</li>
	 * <li>the <i>patch</i> value of version is smaller or equals than the <i>patch</i> value of
	 * compatibilityVersion.</li>
	 * </ul>
	 *
	 * @param compatibilityVersion
	 *            the reference version.
	 * @param version
	 *            the version to check.
	 * @throws IllegalArgumentException
	 *             if the version is not compatible with the compatibilityVersion.
	 */
	public abstract void checkVersion(String compatibilityVersion, String version) throws IllegalArgumentException;

	/**
	 * Disposes given images. This utility method ensures the given images are not null, which allows to simplify caller
	 * code.
	 *
	 * @param images
	 *            the images to dispose.
	 */
	public void disposeIfNotNull(Image... images) {
		if (images != null) {
			for (Image image : images) {
				if (image != null) {
					image.dispose();
				}
			}
		}
	}

}
