/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of a BufferedVectorImage into a BufferedVectorImage.
 */
public class TestBufferedVectorImageMultiple {

	interface Drawer {

		/**
		 * Draws something in graphics context in the specified rectangle.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h);

	}

	interface AlphaDrawer {

		/**
		 * Draws something in graphics context in the specified rectangle and opacity.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 * @param alpha
		 *            the opacity
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h, int alpha);

	}

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Draws a buffered vector image with a path in another buffered vector image
	 */
	@Test
	public static void testDrawPathBVIinBVI() {
		testDrawBVIinBVI(getPathDrawer());
	}

	/**
	 * Draws a buffered vector image with a gradient path in another buffered vector image
	 */
	@Test
	public static void testDrawGradientBVIinBVI() {
		testDrawBVIinBVI(getGradientDrawer());
	}

	/**
	 * Draws a buffered vector image with a raw image in another buffered vector image
	 */
	@Test
	public static void testDrawImageBVIinBVI() {
		testDrawBVIinBVI(getImageDrawer("")); //$NON-NLS-1$
	}

	/**
	 * Draws a buffered vector image with a raw gradient image in another buffered vector image
	 */
	@Test
	public static void testDrawGradientImageBVIinBVI() {
		testDrawBVIinBVI(getImageDrawer("_gradient")); //$NON-NLS-1$
	}

	private static void testDrawBVIinBVI(AlphaDrawer drawer) {

		int sRed = 50;
		int sGreen = 100;
		int left = 50;
		int right = 200;
		int top = 50;
		int bottom = 200;

		int t = 10;

		try (BufferedVectorImage bvi1 = createBufferedVectorImage(sRed, sRed, Colors.RED, drawer,
				GraphicsContext.OPAQUE);
				BufferedVectorImage bvi2 = createBufferedVectorImage(sGreen, sGreen, Colors.GREEN, drawer,
						GraphicsContext.OPAQUE)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();

			// draw both images on gc
			Matrix matrix = new Matrix();
			matrix.setTranslate(left, top);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, top);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			// draw bvi1 in bvi2
			matrix.setTranslate(t, t);
			VectorGraphicsPainter.drawImage(bvi2.getGraphicsContext(), bvi1, matrix);

			// draw both images on gc
			matrix.setTranslate(left, bottom);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, bottom);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			display.flush();

			// test image without modif
			checkAreas("TL", left, top, sRed, sRed, Colors.RED, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("TR", right, top, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image after modif
			checkAreas("BL", left, bottom, sRed, sRed, Colors.RED, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("BR", right, bottom, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image inside image
			checkAreas("BR2", right + t, bottom + t, sRed, sRed, Colors.RED, Colors.GREEN); //$NON-NLS-1$
		}
	}

	/**
	 * Draws a buffered vector image with scaling in another buffered vector image
	 */
	@Test
	public static void testDrawPathScaledBVIinBVI() {
		testDrawScaledBVIinBVI(getPathDrawer());
	}

	/**
	 * Draws a buffered vector image with scaling in another buffered vector image
	 */
	@Test
	public static void testDrawGradientScaledBVIinBVI() {
		testDrawScaledBVIinBVI(getGradientDrawer());
	}

	/**
	 * Draws a buffered vector image with a raw image with scaling in another buffered vector image
	 */
	@Test
	public static void testDrawImageScaledBVIinBVI() {
		testDrawScaledBVIinBVI(getImageDrawer("")); //$NON-NLS-1$
	}

	/**
	 * Draws a buffered vector image with a raw gradient image with scaling in another buffered vector image
	 */
	@Test
	public static void testDrawGradientImageScaledBVIinBVI() {
		testDrawScaledBVIinBVI(getImageDrawer("_gradient")); //$NON-NLS-1$
	}

	private static void testDrawScaledBVIinBVI(AlphaDrawer drawer) {

		int sRed = 50;
		int sGreen = 100;
		int left = 50;
		int right = 200;
		int top = 50;
		int bottom = 200;

		int t = 10;

		try (BufferedVectorImage bvi1 = createBufferedVectorImage(sRed, sRed, Colors.RED, drawer,
				GraphicsContext.OPAQUE);
				BufferedVectorImage bvi2 = createBufferedVectorImage(sGreen, sGreen, Colors.GREEN, drawer,
						GraphicsContext.OPAQUE)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();

			// draw both images on gc
			Matrix matrix = new Matrix();
			matrix.setTranslate(left, top);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, top);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			// draw bvi2 in bvi1
			Matrix matrix2 = new Matrix();
			matrix2.preTranslate(t, t);
			matrix2.preScale(0.25f, 0.25f); // 25 %
			VectorGraphicsPainter.drawImage(bvi1.getGraphicsContext(), bvi2, matrix2);

			// draw both images on gc
			matrix.setTranslate(left, bottom);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, bottom);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			display.flush();

			// test image without modif
			checkAreas("TL", left, top, sRed, sRed, Colors.RED, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("TR", right, top, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image after modif
			checkAreas("BL", left, bottom, sRed, sRed, Colors.RED, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("BR", right, bottom, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image inside image
			checkAreas("BL2", left + t, bottom + t, sGreen / 4, sGreen / 4, Colors.GREEN, Colors.RED); //$NON-NLS-1$
		}
	}

	/**
	 * Draws a buffered vector image in another buffered vector image and clear source image
	 */
	@Test
	public static void testDrawPathBVIinBVIAndClear() {
		testDrawBVIinBVIAndClear(getPathDrawer());
	}

	/**
	 * Draws a buffered vector image in another buffered vector image and clear source image
	 */
	@Test
	public static void testDrawGradientBVIinBVIAndClear() {
		testDrawBVIinBVIAndClear(getGradientDrawer());
	}

	/**
	 * Draws a buffered vector image with a raw image in another buffered vector image and clear source image
	 */
	@Test
	public static void testDrawImageBVIinBVIAndClear() {
		testDrawBVIinBVIAndClear(getImageDrawer("")); //$NON-NLS-1$
	}

	/**
	 * Draws a buffered vector image with a raw gradient image in another buffered vector image and clear source image
	 */
	@Test
	public static void testDrawGradientImageBVIinBVIAndClear() {
		testDrawBVIinBVIAndClear(getImageDrawer("_gradient")); //$NON-NLS-1$
	}

	private static void testDrawBVIinBVIAndClear(AlphaDrawer drawer) {

		int sRed = 50;
		int sGreen = 100;
		int left = 50;
		int right = 200;
		int top = 50;
		int bottom = 200;

		int t = 10;

		try (BufferedVectorImage bvi1 = createBufferedVectorImage(sRed, sRed, Colors.RED, drawer,
				GraphicsContext.OPAQUE);
				BufferedVectorImage bvi2 = createBufferedVectorImage(sGreen, sGreen, Colors.GREEN, drawer,
						GraphicsContext.OPAQUE)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();

			// draw both images on gc
			Matrix matrix = new Matrix();
			matrix.setTranslate(left, top);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, top);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			// draw bvi1 in bvi2
			matrix.setTranslate(t, t);
			VectorGraphicsPainter.drawImage(bvi2.getGraphicsContext(), bvi1, matrix);
			bvi1.clear();

			// draw both images on gc
			matrix.setTranslate(left, bottom);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, bottom);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			display.flush();

			// test image without modif
			checkAreas("TL", left, top, sRed, sRed, Colors.RED, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("TR", right, top, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image after modif
			checkAreas("BL", left, bottom, sRed, sRed, TestUtilities.BACKGROUND_COLOR, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("BR", right, bottom, sGreen, sGreen, Colors.GREEN, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image inside image
			checkAreas("BR2", right + t, bottom + t, sRed, sRed, Colors.RED, Colors.GREEN); //$NON-NLS-1$
		}

	}

	/**
	 * Draws a buffered vector image in another buffered vector image with transparency
	 */
	@Test
	public static void testDrawTransparentPathBVIinBVI() {
		testDrawTransparentBVIinBVI(getPathDrawer());
	}

	/**
	 * Draws a buffered vector image in another buffered vector image with transparency
	 */
	@Test
	public static void testDrawTransparentGradientBVIinBVI() {
		testDrawTransparentBVIinBVI(getGradientDrawer());
	}

	/**
	 * Draws a buffered vector image with a raw image in another buffered vector image with transparency
	 */
	@Test
	public static void testDrawTransparentImageBVIinBVI() {
		testDrawTransparentBVIinBVI(getImageDrawer("")); //$NON-NLS-1$
	}

	/**
	 * Draws a buffered vector image with a raw gradient image in another buffered vector image with transparency
	 */
	@Test
	public static void testDrawTransparentGradientImageBVIinBVI() {
		testDrawTransparentBVIinBVI(getImageDrawer("_gradient")); //$NON-NLS-1$
	}

	private static void testDrawTransparentBVIinBVI(AlphaDrawer drawer) {

		int sRed = 50;
		int sGreen = 100;
		int left = 50;
		int right = 200;
		int top = 50;
		int bottom = 200;

		// expected transparent colors when opacity is 0x80
		int transparentRed = 0x780000;
		int transparentGreen = 0x004000;
		int transparentRedGreen = 0x782000;

		int t = 10;

		try (BufferedVectorImage bvi1 = createBufferedVectorImage(sRed, sRed, Colors.RED, drawer, 0x80);
				BufferedVectorImage bvi2 = createBufferedVectorImage(sGreen, sGreen, Colors.GREEN, drawer, 0x80)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();

			// draw both images on gc
			Matrix matrix = new Matrix();
			matrix.setTranslate(left, top);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, top);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			// draw bvi1 in bvi2
			matrix.setTranslate(t, t);
			VectorGraphicsPainter.drawImage(bvi2.getGraphicsContext(), bvi1, matrix);

			// draw both images on gc
			matrix.setTranslate(left, bottom);
			VectorGraphicsPainter.drawImage(g, bvi1, matrix);
			matrix.setTranslate(right, bottom);
			VectorGraphicsPainter.drawImage(g, bvi2, matrix);

			display.flush();

			// test image without modif
			checkAreas("TL", left, top, sRed, sRed, transparentRed, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("TR", right, top, sGreen, sGreen, transparentGreen, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image after modif
			checkAreas("BL", left, bottom, sRed, sRed, transparentRed, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$
			checkAreas("BR", right, bottom, sGreen, sGreen, transparentGreen, TestUtilities.BACKGROUND_COLOR); //$NON-NLS-1$

			// test image inside image
			checkAreas("BR2", right + t, bottom + t, sRed, sRed, transparentRedGreen, transparentGreen); //$NON-NLS-1$
		}

	}

	private static void checkAreas(String msg, int x, int y, int w, int h, int colorInside, int colorOutside) {
		int errors = 0;
		errors += TestUtilities.checkPeripheralArea(msg + " clip over inside", colorInside, x + 1, y + 1, w - 2, h - 2, //$NON-NLS-1$
				1, 0, false);
		errors += TestUtilities.checkPeripheralArea(msg + " clip over outside", colorOutside, x, y, w, h, 1, 0, false); //$NON-NLS-1$
		assertEquals(0, errors);
	}

	private static BufferedVectorImage createBufferedVectorImage(int w, int h, int color, AlphaDrawer drawer,
			int alpha) {
		BufferedVectorImage image = new BufferedVectorImage(w, h);
		GraphicsContext g = image.getGraphicsContext();
		g.setColor(color);
		drawer.draw(g, 0, 0, w, h, alpha);
		return image;
	}

	private static AlphaDrawer getPathDrawer() {
		return new AlphaDrawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				VectorGraphicsPainter.fillPath(g, getPath(w, h), getTranslatedMatrix(x, y), FillType.WINDING, alpha,
						BlendMode.SRC_OVER);
			}
		};
	}

	private static AlphaDrawer getGradientDrawer() {
		return new AlphaDrawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				int color = g.getColor() | (GraphicsContext.OPAQUE << 24);
				LinearGradient gradient = new LinearGradient(0, 0, w - 1, h - 1, new int[] { color, color, color });
				VectorGraphicsPainter.fillGradientPath(g, getPath(w, h), getTranslatedMatrix(x, y), gradient,
						FillType.WINDING, alpha, BlendMode.SRC_OVER);
			}
		};
	}

	private static AlphaDrawer getImageDrawer(final String suffix) {
		return new AlphaDrawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h, int alpha) {
				int color = g.getColor();
				VectorImage image;
				switch (color) {
				case Colors.RED:
					image = getImage("red" + suffix); //$NON-NLS-1$
					break;
				case Colors.GREEN:
					image = getImage("green" + suffix); //$NON-NLS-1$
					break;
				default:
					Assert.fail("invalid color: 0x" + Integer.toHexString(color)); //$NON-NLS-1$
					return;
				}
				Matrix matrix = getTranslatedMatrix(x, y);
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				VectorGraphicsPainter.drawImage(g, image, matrix, alpha);
			}
		};
	}

	private static Matrix getTranslatedMatrix(int x, int y) {
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		return matrix;
	}

	private static Path getPath(int w, int h) {
		Path path = new Path();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		return path;
	}

	private static VectorImage getImage(String name) {
		return VectorImage.getImage("/com/microej/microvg/test/test_buffered_vector_image_multiple_" + name + ".xml"); //$NON-NLS-1$//$NON-NLS-2$
	}

	/**
	 * Tests bvi image with clip on destination
	 *
	 * @see TestBufferedVectorImageClip#testImageClipOnDestination()
	 */
	@Test
	public static void testDrawBVIInBVIWithClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h);
				BufferedVectorImage source = createRedImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 20 x 20
			// dest: 10 x 10 at (201,201)
			checkAreas(x, y, w, h);
		}
	}

	/**
	 * Tests bvi image with clip on source
	 *
	 * @see TestBufferedVectorImageClip#testImageClipOnDestination()
	 */
	@Test
	public static void testDrawBVIInBVIWithClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20);
				BufferedVectorImage source = createRedImage(20, 20)) {

			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setColor(Colors.RED);
			ig.setClip(2, 2, 5, 5);
			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 5 x 5
			// dest: 5 x 5 at (202,202)
			checkAreas(2, 2, 5, 5);
		}
	}

	/**
	 * Tests bvi image with clip on source and destination
	 *
	 * @see TestBufferedVectorImageClip#testImageClipOnDestination()
	 */
	@Test
	public static void testDrawBVIInBVIWithClipOnSourceAndDestination() {

		int x = 1;
		int y = 10;
		int w = 25;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60);
				BufferedVectorImage source = createRedImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setClip(5, 5, 10, 10);
			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			VectorGraphicsPainter.drawImage(g, image, new Matrix());
			display.flush();

			// source: 10 x 10
			// dest: 10 x 5 at (205,210)
			checkAreas(5, y, 10, 5);
		}
	}

	/**
	 * Tests bvi image with scaling on source
	 *
	 * @see TestBufferedVectorImageClip#testImageScalingOnSource()
	 */
	@Test
	public static void testDrawBVIInBVIWithScalingOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(20, 20);
				BufferedVectorImage source = createRedImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 5 x 5 at (200,200)
			checkAreas(0, 0, 5, 5);
		}
	}

	/**
	 * Tests bvi image with scaling on source and clip on destination
	 *
	 * @see TestBufferedVectorImageClip#testImageScalingOnSourceAndClipOnDestination()
	 */
	@Test
	public static void testDrawBVIInBVIWithScalingOnSourceAndClipOnDestination() {

		int x = 1;
		int y = 1;
		int w = 10;
		int h = 10;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(2 * w, 2 * h);
				BufferedVectorImage source = createRedImage(20, 20)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 20 x 20
			// dest: 4 x 4 at (201,201)
			checkAreas(x, y, 4, 4);
		}
	}

	/**
	 * Tests bvi image with scaling and clip on source
	 *
	 * @see TestBufferedVectorImageClip#testImageScalingAndClipOnSource()
	 */
	@Test
	public static void testDrawBVIInBVIWithScalingAndClipOnSource() {

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60);
				BufferedVectorImage source = createRedImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);

			ig.setClip(12, 12, 32, 32);
			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 5 x 5 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 8, 8);
		}
	}

	/**
	 * Tests bvi image with scaling on source and clip on source and destination
	 *
	 * @see TestBufferedVectorImageClip#testImageScalingOnSourceAndClipOnSourceAndDestination()
	 */
	@Test
	public static void testDrawBVIInBVIWithScalingOnSourceAndClipOnSourceAndDestination() {

		int x = 1;
		int y = 1;
		int w = 5;
		int h = 5;

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		try (BufferedVectorImage image = new BufferedVectorImage(60, 60);
				BufferedVectorImage source = createRedImage(60, 60)) {
			GraphicsContext ig = image.getGraphicsContext();

			g.translate(200, 200);
			g.setClip(x, y, w, h);

			ig.setClip(12, 12, 32, 32);
			VectorGraphicsPainter.drawImage(ig, source, new Matrix());

			Matrix matrix = new Matrix();
			matrix.setScale(0.25f, 0.25f);
			VectorGraphicsPainter.drawImage(g, image, matrix);
			display.flush();

			// source: 60 x 60
			// dest: 3 x 3 at (203,203) // 3 = 12 / 4
			checkAreas(3, 3, 3, 3);
		}
	}

	private static void checkAreas(int x, int y, int w, int h) {
		int errors = 0;
		errors += TestUtilities.checkPeripheralArea("clip over inside", Colors.RED, x + 1, y + 1, w - 2, h - 2, 1, 0, //$NON-NLS-1$
				false);
		errors += TestUtilities.checkPeripheralArea("clip over outside", TestUtilities.BACKGROUND_COLOR, x, y, w, h, 1, //$NON-NLS-1$
				0, false);
		assertEquals(0, errors);
	}

	private static BufferedVectorImage createRedImage(int w, int h) {
		BufferedVectorImage image = new BufferedVectorImage(w, h);
		GraphicsContext ig = image.getGraphicsContext();
		ig.setColor(Colors.RED);
		Path path = new Path();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		VectorGraphicsPainter.fillPath(ig, path, 0, 0);
		return image;
	}
}
