/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector.combinator;

import ej.mwt.Container;
import ej.mwt.Widget;
import ej.mwt.stylesheet.selector.Selector;

/**
 * A general sibling combinator selects by checking the widget sibling.
 * <p>
 * Equivalent to <code>widget~widget</code> selector in CSS.
 *
 * @see Widget#getParent()
 */
public class GeneralSiblingCombinator extends Combinator {

	/**
	 * Creates a general sibling combinator.
	 *
	 * @param selector1
	 *            the first selector.
	 * @param selector2
	 *            the second selector.
	 */
	public GeneralSiblingCombinator(Selector selector1, Selector selector2) {
		super(selector1, selector2);
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		// check that second selector applies to widget
		if (!getSecondSelector().appliesToWidget(widget)) {
			return false;
		}

		// get parent
		Container parent = widget.getParent();
		if (parent == null) {
			return false;
		}

		// get widget index in parent
		int index = parent.getChildIndex(widget);

		while (index > 0) {
			// check that first selector applies to previous sibling widget
			Widget siblingWidget = parent.getChild(index - 1);
			if (getFirstSelector().appliesToWidget(siblingWidget)) {
				return true;
			}

			// try previous sibling
			index--;
		}

		return false;
	}
}
