/*
 * <Java>
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.service;

import java.security.Permission;

import ej.annotation.Nullable;

/**
 * Permission to manipulate services using {@link ServiceLoader#getService(Class)} or
 * {@link ServiceRegistry#register(Class, Object)} or {@link ServiceRegistry#unregister(Class, Object)}.
 */
public class ServicePermission extends Permission {

	private static final long serialVersionUID = -4972653329773764089L;

	/**
	 * Action associated with {@link ServiceLoader#getService(Class)}.
	 */
	public static final String GET_ACTION = "get"; //$NON-NLS-1$
	/**
	 * Action associated with {@link ServiceRegistry#register(Class, Object)}.
	 */
	public static final String REGISTER_ACTION = "register"; //$NON-NLS-1$
	/**
	 * Action associated with {@link ServiceRegistry#unregister(Class, Object)}.
	 */
	public static final String UNREGISTER_ACTION = "unregister"; //$NON-NLS-1$

	private final Class<?> service;
	private final String action;

	/**
	 * Creates a permission for the given service.
	 * <p>
	 * The action can be one of {@link #GET_ACTION}, {@link #REGISTER_ACTION} or {@link #UNREGISTER_ACTION}.
	 *
	 * @param parentName
	 *            the name of the service loader or service registry.
	 * @param service
	 *            the service.
	 * @param action
	 *            the action realized with the service.
	 */
	public ServicePermission(String parentName, Class<?> service, String action) {
		super(parentName + ':' + service.getName());
		this.service = service;
		this.action = action;
	}

	@Override
	public String getActions() {
		return this.action;
	}

	/**
	 * Gets the service handled by this permission.
	 *
	 * @return the service.
	 */
	public Class<?> getService() {
		return this.service;
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		return this == obj;
	}

	@Override
	public int hashCode() {
		return System.identityHashCode(this);
	}

	@Override
	public boolean implies(@Nullable Permission permission) {
		return equals(permission);
	}

}
