/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Image;
import ej.microui.display.Painter;
import ej.microui.display.ResourceImage;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests kerning.
 */
@SuppressWarnings("nls")
public class TestFontComplexLayout {

	private static final String AMIRI_FONT = "/fonts/Amiri-reduced.ttf";
	private static final String NOTOSANSARABIC_FONT = "/fonts/NotoSansArabic-reduced.ttf";
	private static final String NOTOSERIFTHAI_FONT = "/fonts/NotoSerifThai-Regular.ttf";

	private static final String ARABIC_STRING = "الطريق أخضر ومظلل";
	private static final String THAI_STRING = "ผีเสื้อสีแดงกำลังโบยบิน";
	private static final String[] ARABIC_STRINGS = { "وكان معه المساع", "الذي يحمل له", "عدة التصوير،",
			"ويظهر في الصور", "كيف كان شكل الكاميرا" };

	private static final String[] THAI_STRINGS = { "ลิงกับแมวเหมือนช้าง", "ส่วนที่ดีที่สุดในชีวิตฉัน",
			"ลูกสาวของฉันเป็นหมอฟัน", "จะไปประเทศใครบ้าง", "หมาแดงอยู่ในสระว่ายน้ำ" };

	/**
	 * Tolerance for the density test.
	 * <p>
	 * Represents the average difference per pixel.
	 *
	 * @see #checkString(String, int, int, int, int)
	 */
	private static final float TOLERANCE_DENSITY = 0.04f;
	/**
	 * Tolerance for the bumps test.
	 * <p>
	 * Represents the average difference of bumps by line.
	 *
	 * @see #checkString(String, int, int, int, int)
	 */
	private static final float TOLERANCE_BUMPS = 1.1f;
	/**
	 * Tolerance for the Levenshtein distance test.
	 * <p>
	 * Represents the average difference of distance by line.
	 *
	 * @see #checkString(String, int, int, int, int)
	 */
	private static final float TOLERANCE_LEVENSHTEIN = 4.1f;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing an arabic string with amiri font without complex layout.
	 */
	@Test
	public void testDrawStringLayoutArabicAmiri() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT);
		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = 50;
		int y = 50;

		checkString("testDrawStringLayoutArabic_amiri_simple", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing an arabic string with amiri font with complex layout.
	 */
	@Test
	public void testDrawStringLayoutArabicAmiriComplex() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);
		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = 50;
		int y = 50;

		checkString("testDrawStringLayoutArabic_amiri_complex", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing an arabic string with another arabic font without complex layout.
	 */
	@Test
	public void testDrawStringLayoutArabicNotoArabic() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSANSARABIC_FONT);
		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = 50;
		int y = 50;

		checkString("testDrawStringLayoutArabic_notoarabic_simple", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing an arabic string with another arabic font with complex layout.
	 */
	@Test
	public void testDrawStringLayoutArabicNotoArabicComplex() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSANSARABIC_FONT, true);
		String string = ARABIC_STRING;

		int fontSize = 30;
		int x = 50;
		int y = 50;

		checkString("testDrawStringLayoutArabic_notoarabic_complex", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing a thai string with 1 font without complex layout
	 */
	@Test
	public void testDrawStringLayoutThaiSimple() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSERIFTHAI_FONT);

		String string = THAI_STRING;
		int fontSize = 30;
		int x = 30;

		int y = 50;
		checkString("testDrawStringLayoutThai_notothai_simple", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing a thai string with 1 font with complex layout
	 */
	@Test
	public void testDrawStringLayoutThaiComplex() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSERIFTHAI_FONT, true);

		String string = THAI_STRING;
		int fontSize = 30;
		int x = 30;

		int y = 50;
		checkString("testDrawStringLayoutThai_notothai_complex", g, font, string, fontSize, x, y);
	}

	/**
	 * Test drawing an arabic text
	 *
	 */
	@Test
	public void testDrawStringLayoutArabicText() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(AMIRI_FONT, true);

		g.resetClip();

		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		g.setColor(Colors.RED);
		int xRight = 300;
		Painter.drawLine(g, xRight, 0, xRight, display.getHeight());

		int fontSize = 30;
		g.setColor(Colors.WHITE);
		Matrix m = new Matrix();

		for (int i = 0; i < ARABIC_STRINGS.length; i++) {
			String string = ARABIC_STRINGS[i];
			assert string != null;

			float stringWidth = font.measureStringWidth(string, fontSize);
			int x = (int) (xRight - stringWidth);
			int y = (int) (40 + i * (font.getHeight(fontSize) + 3));

			checkString("testDrawStringLayoutArabic_arabic_" + (i + 1), g, font, string, fontSize, x, y, m);
		}

		display.flush();
	}

	/**
	 * Test drawing an thai text
	 *
	 */
	@Test
	public void testDrawStringLayoutThaiText() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSERIFTHAI_FONT, true);

		g.resetClip();

		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		g.setColor(Colors.RED);
		int xLeft = 30;
		Painter.drawLine(g, xLeft, 0, xLeft, display.getHeight());

		int fontSize = 30;
		g.setColor(Colors.WHITE);
		Matrix m = new Matrix();

		for (int i = 0; i < THAI_STRINGS.length; i++) {
			String string = THAI_STRINGS[i];
			assert string != null;

			int x = (xLeft + 1);
			int y = (int) (40 + i * (font.getHeight(fontSize) + 3));

			checkString("testDrawStringLayoutThai_thai_" + (i + 1), g, font, string, fontSize, x, y, m);
		}

		display.flush();
	}

	/**
	 * Test drawing an arabic string on a circle
	 *
	 */
	@Test
	public void testDrawStringLayoutOnCircle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = VectorFont.loadFont(NOTOSANSARABIC_FONT, true);

		g.resetClip();

		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());

		String string = ARABIC_STRING;

		final int radius = 2000;
		final int fontSize = 30;
		final int x = display.getWidth() / 2;
		final int y = display.getHeight() / 2;

		g.setColor(Colors.WHITE);
		Matrix matrix = new Matrix();
		matrix.setRotate(86);
		matrix.postTranslate(x, y - radius);

		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE);

		final int expectedX = x - 103;
		final int expectedY = y - 13;
		final int expectedW = 244;
		final int expectedH = 42;

		display.flush();
		checkString("testDrawStringLayoutOnCircle_arabic", expectedX, expectedY, expectedW, expectedH);
	}

	private void checkString(String testName, GraphicsContext g, VectorFont font, String string, float fontSize, int x,
			int y, Matrix m) {
		int width = (int) Math.ceil(font.measureStringWidth(string, fontSize));
		int height = (int) Math.ceil(font.getHeight(fontSize));
		g.setClip(x, y, width, height);
		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, x, y, width, height);
		g.setColor(Colors.WHITE);
		drawRectangleCorners(g, x, y, width, height, 4);
		m.setTranslate(x, y);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, m, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);

		checkString(testName, x - 1, y - 1, width + 2, height + 2);
		Display.getDisplay().flush();
	}

	private void checkString(String testName, GraphicsContext g, VectorFont font, String string, float fontSize, int x,
			int y) {
		int width = (int) Math.ceil(font.measureStringWidth(string, fontSize));
		int height = (int) Math.ceil(font.getHeight(fontSize));
		g.setClip(x, y, width, height);
		g.setColor(Colors.BLACK);
		Painter.fillRectangle(g, x, y, width, height);
		g.setColor(Colors.WHITE);
		drawRectangleCorners(g, x, y, width, height, 4);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);

		checkString(testName, x - 1, y - 1, width + 2, height + 2);
		Display.getDisplay().flush();
	}

	/**
	 * Levenshtein distance: https://en.wikipedia.org/wiki/Levenshtein_distance
	 * <p>
	 * Implementations: https://en.wikibooks.org/wiki/Algorithm_Implementation/Strings/Levenshtein_distance#Java (not
	 * recursive used here)
	 */
	private static void checkString(String filename, int xOffset, int yOffset, int widthString, int heightString) {
		GraphicsContext g = Display.getDisplay().getGraphicsContext();

		String imagePath = "/images/" + filename + "_ref.png";
		Image ref = ResourceImage.loadImage(imagePath);

		int width = ref.getWidth();
		int height = ref.getHeight();

		if (height != heightString) {
			int heightDiff = Math.abs(height - heightString);
			float heightDiffRatio = (float) heightDiff / height;
			Assert.assertFalse(heightDiffRatio > 0.01);
		}
		int widthDiff = Math.abs(width - widthString);
		if (width != widthString) {
			float widthDiffRatio = (float) widthDiff / width;
			System.out.println("Width diff: " + widthDiff + " (ratio: " + widthDiffRatio + ")");
			Assert.assertFalse(widthDiffRatio > 0.03);
		}

		// Density variables.
		int totalDensityDiff = 0;

		// Bumps variables.
		int totalBumpsDiff = 0;
		int maxBumpsDiff = 0;

		// Levenshtein distance variables.
		int[] cost = new int[widthString + 1];
		int[] newcost = new int[widthString + 1];
		int[] lineString = new int[widthString];
		int maxLevenshteinDistance = 0;
		int totalLevenshteinDistance = 0;

		for (int y = 0; y < height; y++) {
			int yString = yOffset + y;

			// Line density initialization.
			int densityPixelsRef = 0;
			int densityPixelsString = 0;

			// Line bumps initialization.
			boolean inBumpRef = false;
			int levelRef = 0;
			int previousRef = 0;
			boolean inBumpString = false;
			int levelString = 0;
			int previousString = 0;
			int bumpsRef = 0;
			int bumpsString = 0;

			// Line Levenshtein distance initialization.
			g.readPixels(lineString, 0, widthString, xOffset, yString, widthString, 1);
			for (int x = 0; x <= widthString; x++) {
				cost[x] = x;
			}

			for (int x = 0; x < width; x++) {
				int colorRef = getPixel(ref, x, y);
				if (x < widthString) {
					int colorString = lineString[x] & 0xff;// getPixel(g, xOffset + x, yString);

					// Compute density in reference and current string.
					densityPixelsRef += colorRef;
					densityPixelsString += colorString;

					// Compute bumps in reference and current string.
					levelRef += colorRef;
					if (colorRef == 0x0 || colorRef < previousRef) {
						// At startup or the color level is decreasing.
						levelRef = 0;
						inBumpRef = false;
					} else if (!inBumpRef && levelRef >= 0x80) {
						// New bump detected.
						bumpsRef++;
						inBumpRef = true;
					}
					previousRef = colorRef;
					levelString += colorString;
					if (colorString == 0x0 || colorString < previousString) {
						// At startup or the color level is decreasing.
						levelString = 0;
						inBumpString = false;
					} else if (!inBumpString && levelString >= 0x80) {
						// New bump detected.
						bumpsString++;
						inBumpString = true;
					}
					previousString = colorString;
				}

				// Compute Levenshtein distance between the two lines.
				newcost[0] = x;
				// The analysis is limited to 10 pixels before and 10 after to speed-up the algorithm.
				int start = Math.max(0, x - 10);
				int stop = Math.min(widthString, x + 10);
				for (int xStringT = start; xStringT < stop; xStringT++) {
					// Check if the pixel matches the pixels in the other line.
					boolean match = Math.abs(colorRef - (lineString[xStringT] & 0xff)) < 0x80;

					int costSubstitute = cost[xStringT] + (match ? 0 : 1); // Substitution.
					int costInsert = cost[xStringT + 1] + 1; // Insertion.
					int costDelete = newcost[xStringT] + 1; // Deletion.

					// Keep minimum cost.
					newcost[xStringT + 1] = Math.min(Math.min(costInsert, costDelete), costSubstitute);
				}

				int[] swap = cost;
				cost = newcost;
				newcost = swap;
			}

			// Compute density difference for the line.
			// The density of a line is simply the some of all its pixels.
			int densityDiff = Math.abs(densityPixelsRef - densityPixelsString);
			totalDensityDiff += densityDiff;

			// Compute bumps difference for the line.
			int bumpsDiff = Math.abs(bumpsRef - bumpsString);
			totalBumpsDiff += bumpsDiff;
			maxBumpsDiff = Math.max(maxBumpsDiff, bumpsDiff);

			// Compute total Levenshtein distance.
			// It is the sum of the transformations of each pixel in both lines.
			int levenshteinDistance = cost[widthString];
			totalLevenshteinDistance += levenshteinDistance;
			maxLevenshteinDistance = Math.max(maxLevenshteinDistance, levenshteinDistance);
		}

		System.out.println("Density diff=" + (float) totalDensityDiff / (width * height * 0xff));
		System.out.println("Bumps diff=" + (float) totalBumpsDiff / height);
		System.out.println("Levenshtein Distance=" + ((float) totalLevenshteinDistance / height - widthDiff));

		Assert.assertFalse("Too much density difference",
				(float) totalDensityDiff / (width * height * 0xff) > TOLERANCE_DENSITY);
		Assert.assertFalse("Too many bumps difference", (float) totalBumpsDiff / height > TOLERANCE_BUMPS);
		Assert.assertFalse("Too high levenshtein distance",
				(float) (totalLevenshteinDistance / height) - widthDiff > TOLERANCE_LEVENSHTEIN);
	}

	private static int getPixel(Image image, int x, int y) {
		return image.readPixel(x, y) & 0xff;
	}

	private void drawRectangleCorners(GraphicsContext g, int x, int y, int width, int height, int size) {
		Painter.drawRectangle(g, x, y, size, size);
		Painter.drawRectangle(g, x + width - size, y, size, size);
		Painter.drawRectangle(g, x + width - size, y + height - size, size, size);
		Painter.drawRectangle(g, x, y + height - size, size, size);
	}

}
