/*
 * Java
 *
 * Copyright 2009-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt;

import ej.annotation.NonNull;
import ej.microui.display.Display;
import ej.microui.display.DisplayPump;

/**
 * A dialog is a specific panel that once shown is the only one that receives all input events until it is hidden.
 * <p>
 * The show methods ({@link #showFullScreen(Desktop)}, {@link #showAdjustingToChild(Desktop)},
 * {@link #showUsingBounds(Desktop)}) do not return while the dialog is shown. During this period, all the input events
 * are sent only to this dialog and its children. Beware that when opening a dialog:
 * <ul>
 * <li>If the current thread is used to read data or to communicate, these actions will not run until the dialog is
 * hidden.</li>
 * <li>If the current thread owns monitors, they will not be released. Other threads that need these monitors will wait
 * until the dialog is hidden.</li>
 * </ul>
 * <p>
 * If several dialogs are shown, they are stacked and only the last one is active.
 */
public class Dialog extends Panel {

	/**
	 * Creates a new dialog.
	 */
	public Dialog() {
		super();
	}

	@Override
	protected void showInternal(@NonNull Desktop desktop) {
		super.showInternal(desktop);
		Display display = desktop.getDisplay();
		if (display.isDisplayThread()) {
			// copied from DisplayPump.run()
			DisplayPump pump = display.pump;
			while (pump.isRunning() && isShown()) {
				pump.pump();
			}
		} else {
			synchronized (this) {
				while (this.isShown()) {
					try {
						this.wait();
					} catch (InterruptedException e) {
						// Nothing to do.
					}
				}
			}
		}
	}

	@Override
	/* package */void showToDesktop(@NonNull Desktop desktop) {
		desktop.addDialog(this);
	}

	@Override
	/* package */void hideFromDesktop(@NonNull Desktop desktop) {
		desktop.removeDialog(this);
		synchronized (this) {
			detachFromDesktop();
			this.notifyAll();
		}
	}

}
