/*
 * Java
 *
 * Copyright 2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.util;

import ej.annotation.NonNull;
import ej.annotation.Nullable;

/**
 * Refer to <a href="https://www.bluetooth.com/specifications/assigned-numbers/generic-access-profile">Bluetooth
 * Assigned Numbers</a>.
 */
public class BluetoothPayloadUtil {

	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.3 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.3 and 18.1 (v4.0)Core Specification Supplement, Part A, section 1.3.
	 */
	public static final byte FLAGS = 0x01;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.1 and 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID16_PARTIAL_LIST = 0x02;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.1 and 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID16_FULL_LIST = 0x03;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, section
	 * 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID32_PARTIAL_LIST = 0x04;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, section
	 * 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID32_FULL_LIST = 0x05;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.1 and 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID128_PARTIAL_LIST = 0x06;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.1 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.1 and 18.2 (v4.0)Core Specification Supplement, Part A, section 1.1.
	 */
	public static final byte SERVICE_UUID128_FULL_LIST = 0x07;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.2 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.2 and 18.4 (v4.0)Core Specification Supplement, Part A, section 1.2.
	 */
	public static final byte SHORTENED_LOCAL_NAME = 0x08;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.2 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.2 and 18.4 (v4.0)Core Specification Supplement, Part A, section 1.2.
	 */
	public static final byte COMPLETE_LOCAL_NAME = 0x09;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.5 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.5 and 18.3 (v4.0)Core Specification Supplement, Part A, section 1.5.
	 */
	public static final byte TX_POWER_LEVEL = 0x0A;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, sections 11.1.8 and 18.8 (v4.0)Core Specification Supplement, Part
	 * A, section 1.9.
	 */
	public static final byte CONNECTION_INTERVAL = 0x12;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, sections 11.1.9 and 18.9 (v4.0)Core Specification Supplement, Part
	 * A, section 1.10.
	 */
	public static final byte SOLICITATION_UUID16_LIST = 0x14;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, sections 11.1.9 and 18.9 (v4.0)Core Specification Supplement, Part
	 * A, section 1.10.
	 */
	public static final byte SOLICITATION_UUID128_LIST = 0x15;
	/**
	 * Core Specification Supplement, Part A, section 1.11.
	 */
	public static final byte SERVICE_DATA_UUID16 = 0x16;
	/**
	 * Bluetooth Core Specification:Core Specification Supplement, Part A, section 1.13.
	 */
	public static final byte PUBLIC_TARGET_ADDRESS = 0x17;
	/**
	 * Bluetooth Core Specification:Core Specification Supplement, Part A, section 1.14.
	 */
	public static final byte RANDOM_TARGET_ADDRESS = 0x18;
	/**
	 * Bluetooth Core Specification:Core Specification Supplement, Part A, section 1.12.
	 */
	public static final byte APPEARANCE = 0x19;
	/**
	 * Bluetooth Core Specification:Core Specification Supplement, Part A, section 1.15.
	 */
	public static final byte ADVERTISING_INTERVAL = 0x1A;
	/**
	 * Core Specification Supplement, Part A, section 1.10.
	 */
	public static final byte SOLICITATION_UUID32_LIST = 0x1F;
	/**
	 * Core Specification Supplement, Part A, section 1.11.
	 */
	public static final byte SERVICE_DATA_UUID32 = 0x20;
	/**
	 * Core Specification Supplement, Part A, section 1.11.
	 */
	public static final byte SERVICE_DATA_UUID128 = 0x21;
	/**
	 * Bluetooth Core Specification:Core Specification Supplement, Part A, section 1.18.
	 */
	public static final byte URI = 0x24;
	/**
	 * Bluetooth Core Specification:Vol. 3, Part C, section 8.1.4 (v2.1 + EDR, 3.0 + HS and 4.0)Vol. 3, Part C, sections
	 * 11.1.4 and 18.11 (v4.0)Core Specification Supplement, Part A, section 1.4.
	 */
	public static final byte MANUFACTURER_DATA = (byte) 0xFF;

	/**
	 * LE Limited Discoverable Mode.
	 */
	public static final byte FLAG_LE_LIMITED_DISC_MODE = 0x01;
	/**
	 * LE General Discoverable Mode.
	 */
	public static final byte FLAG_LE_GENERAL_DISC_MODE = 0x02;
	/**
	 * BR/EDR not supported.
	 */
	public static final byte FLAG_BR_EDR_NOT_SUPPORTED = 0x04;
	/**
	 * Simultaneous LE and BR/EDR, Controller.
	 */
	public static final byte FLAG_LE_BR_EDR_CONTROLLER = 0x08;
	/**
	 * Simultaneous LE and BR/EDR, Host.
	 */
	public static final byte FLAG_LE_BR_EDR_HOST = 0x10;

	/**
	 * LE Limited Discoverable Mode, BR/EDR not supported.
	 */
	public static final byte FLAGS_LE_ONLY_LIMITED_DISC_MODE = (FLAG_LE_LIMITED_DISC_MODE | FLAG_BR_EDR_NOT_SUPPORTED);

	/**
	 * LE General Discoverable Mode, BR/EDR not supported.
	 */
	public static final byte FLAGS_LE_ONLY_GENERAL_DISC_MODE = (FLAG_LE_GENERAL_DISC_MODE | FLAG_BR_EDR_NOT_SUPPORTED);

	private BluetoothPayloadUtil() {
		// private constructor
	}

	/**
	 * Gets the value of a type from the payload.
	 *
	 * @param type
	 *            the type.
	 * @param payload
	 *            the payload.
	 * @return the bytes of the type, <code>null</code> if the type has not been found.
	 */
	public static byte[] getBytes(byte type, @NonNull byte[] payload) {
		int index = getIndex(type, payload);
		if (index > -1) {
			int length = payload[index] - 1;
			byte[] data = new byte[length];
			System.arraycopy(payload, index + 2, data, 0, length);
			return data;
		}
		return null;
	}

	/**
	 * Gets the string value of a type from the payload.
	 *
	 * @param type
	 *            the type.
	 * @param payload
	 *            the payload.
	 * @return the value of the type, <code>null</code> if the type has not been found.
	 */
	public static String getString(byte type, @NonNull byte[] payload) {
		byte[] data = getBytes(type, payload);
		if (data == null) {
			return null;
		}
		return new String(data);
	}

	/**
	 * Gets the byte of a type from the payload.
	 *
	 * @param type
	 *            the type.
	 * @param payload
	 *            the payload.
	 * @return the value of the type, <code>null</code> if the type has not been found or its length is more than 1.
	 */
	public static Byte getByte(byte type, @NonNull byte[] payload) {
		byte[] data = getBytes(type, payload);
		if (data == null || data.length != 1) {
			return null;
		}
		return Byte.valueOf(data[0]);
	}

	/**
	 * Appends a new type to a payload. This does not check if the type already exists.
	 *
	 * @param type
	 *            the new type to append.
	 * @param value
	 *            the bytes to append.
	 * @param payload
	 *            the payload to append to.
	 * @return a new array with the current payload plus the length of the next type, the type and type value,
	 */
	public static byte[] appendPayload(byte type, @NonNull byte[] value, @Nullable byte[] payload) {
		int arrayLength = payload.length + value.length + 2; // can throw NullPointerException
		byte[] result = new byte[arrayLength];
		System.arraycopy(payload, 0, result, 0, payload.length);
		result[payload.length] = (byte) ((value.length + 1) & 0xFF);
		result[payload.length + 1] = type;
		System.arraycopy(value, 0, result, payload.length + 2, value.length);
		return result;
	}

	/**
	 * Remove a type from the payload.
	 *
	 * @param type
	 *            the type to remove.
	 * @param payload
	 *            the payload to handle.
	 * @return a new payload without the type of it has been found, the payload if not found.
	 */
	public static byte[] removePayload(byte type, byte[] payload) {
		int index = getIndex(type, payload);
		if (index > -1) {
			byte[] newPayload = new byte[payload.length - payload[index] - 1];
			if (index > 0) {
				System.arraycopy(payload, 0, newPayload, 0, index - 1);
			}
			int nextPart = index + payload[index] + 1;
			if (nextPart < payload.length) {
				System.arraycopy(payload, nextPart, newPayload, index, payload.length - nextPart);
			}
			payload = newPayload;
		}
		return payload;
	}

	/**
	 * Gets the index of the start of a type.
	 *
	 * @param type
	 *            the type to seek.
	 * @param payload
	 *            the payload to look from.
	 * @return the index.
	 */
	private static int getIndex(byte type, byte[] payload) {
		int index = -1;
		int i = 0;
		while (i < payload.length) {
			int length = payload[i];
			// + 1 for the length, + 1 for the type
			int nextIndex = i + length + 1;
			if (length == 0 || nextIndex - 1 >= payload.length) {
				break;
			}

			// byte currenTtype = (byte) ByteArray.readUnsignedByte(payload, i + 1);
			byte currenTtype = (byte) (payload[i + 1] & 0xFF);
			if (currenTtype == type) {
				index = i;
				break;
			}
			i = nextIndex;
		}
		return index;
	}
}
