/*
 * Java
 *
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microvg.image;

import java.awt.Color;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.Arrays;

import ej.microvg.LLVGConstants;
import ej.microvg.MatrixHelper;
import ej.microvg.image.pathdata.PathData;
import ej.microvg.image.pathdata.SingleArrayPathDataFP32;
import ej.microvg.image.pathdata.SingleArrayPathDataS16;
import ej.microvg.image.pathdata.SingleArrayPathDataS32;
import ej.microvg.image.pathdata.SingleArrayPathDataS8;
import ej.microvg.paint.LinearGradient;

/**
 * The VGLite engine.
 */
public class VgliteImageDecoder implements ImageDecoder {

	/**
	 * The path data formats.
	 */
	private static final int VG_LITE_S8 = 0;
	private static final int VG_LITE_S16 = 1;
	private static final int VG_LITE_S32 = 2;
	private static final int VG_LITE_FP32 = 3;

	/**
	 * The path commands.
	 */
	private static final int CMD_CLOSE = 0;
	private static final int CMD_MOVE = 2;
	private static final int CMD_MOVE_REL = 3;
	private static final int CMD_LINE = 4;
	private static final int CMD_LINE_REL = 5;
	private static final int CMD_QUAD = 6;
	private static final int CMD_QUAD_REL = 7;
	private static final int CMD_CUBIC = 8;
	private static final int CMD_CUBIC_REL = 9;

	/**
	 * The fill rules.
	 */
	@SuppressWarnings("unused")
	private static final int VG_LITE_FILL_NON_ZERO = 0;
	private static final int VG_LITE_FILL_EVEN_ODD = 1;

	@Override
	public PathData decodePath(ByteBuffer is) {

		// skip the bounding box (useless for AWT)
		is.getFloat(); // bounding box 0
		is.getFloat(); // bounding box 1
		is.getFloat(); // bounding box 2
		is.getFloat(); // bounding box 3

		// retrieve the path array
		int length = is.getInt();
		byte format = is.get();

		// padding
		is.get();
		is.get();
		is.get();

		ByteBuffer buffer = ByteBuffer.wrap(is.array(), is.position(), length).order(ByteOrder.LITTLE_ENDIAN);
		PathData parser;

		switch (format) {
		case VG_LITE_S8:
			parser = new SingleArrayPathDataS8(buffer);
			break;
		case VG_LITE_S16:
			parser = new SingleArrayPathDataS16(buffer);
			break;
		case VG_LITE_S32:
			parser = new SingleArrayPathDataS32(buffer);
			break;
		case VG_LITE_FP32:
			parser = new SingleArrayPathDataFP32(buffer);
			break;
		default:
			throw new IllegalArgumentException("Unkown format: " + format);
		}

		return parser;
	}

	@Override
	public LinearGradient decodeGradient(ByteBuffer is) {
		final int GRADIENT_VLC_MAX_GRAD = 16; // VGLite lib value
		final int GRADIENT_MAX_STOPS = 255; // VGLite lib value

		// read colors
		Color[] colors = new Color[GRADIENT_VLC_MAX_GRAD];
		for (int i = 0; i < GRADIENT_VLC_MAX_GRAD; i++) {
			colors[i] = new Color(is.getInt(), true);
		}

		// read count
		int count = is.getInt();

		// read positions
		float[] positions = new float[GRADIENT_VLC_MAX_GRAD];
		for (int i = 0; i < GRADIENT_VLC_MAX_GRAD; i++) {
			double pos = is.getInt();
			pos /= GRADIENT_MAX_STOPS;
			positions[i] = (float) pos;
		}

		// read matrix (unscale the gradient to obtain positions values between 0 and 1)
		float[] matrix = getMatrix(is);
		MatrixHelper.scale(matrix, GRADIENT_MAX_STOPS + 1f, 1f);

		return new LinearGradient(Arrays.copyOf(colors, count), Arrays.copyOf(positions, count), matrix);
	}

	@Override
	public int decodeCommand(int encodedCommand) {
		switch (encodedCommand) {
		default: // unknown -> close (should not occur)
		case CMD_CLOSE:
			return LLVGConstants.COMMAND_CLOSE;
		case CMD_MOVE:
			return LLVGConstants.COMMAND_MOVE;
		case CMD_MOVE_REL:
			return LLVGConstants.COMMAND_MOVE_REL;
		case CMD_LINE:
			return LLVGConstants.COMMAND_LINE;
		case CMD_LINE_REL:
			return LLVGConstants.COMMAND_LINE_REL;
		case CMD_QUAD:
			return LLVGConstants.COMMAND_QUAD;
		case CMD_QUAD_REL:
			return LLVGConstants.COMMAND_QUAD_REL;
		case CMD_CUBIC:
			return LLVGConstants.COMMAND_CUBIC;
		case CMD_CUBIC_REL:
			return LLVGConstants.COMMAND_CUBIC_REL;
		}
	}

	@Override
	public int decodeFillRule(int rule) {
		return (VG_LITE_FILL_EVEN_ODD == rule) ? LLVGConstants.FILLTYPE_EVEN_ODD : LLVGConstants.FILLTYPE_WINDING;
	}

	private static float[] getMatrix(ByteBuffer is) {
		float[] matrix = MatrixHelper.create();
		for (int i = 0; i < matrix.length; i++) {
			matrix[i] = is.getFloat();
		}
		return matrix;
	}
}
