/*
 * Java
 *
 * Copyright 2022-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

import ej.fp.Image;

/**
 * Provides a service to create an object linked to a MicroUI BufferedImage.
 * <p>
 * This kind of image is created when a MicroUI extension library offers the possibility to create a custom MicroUI
 * BufferedImage with a specific image format. This image may not hold a pixels buffer.
 * <p>
 * The Platform's Front Panel project (or a library) has to mention the implementation of this interface in a META-INF
 * file: <code>[src folder]/META-INF/services/ej.microui.display.BufferedImageProvider</code>:
 *
 * <pre>
 * ej.xxx.XXXProvider
 * </pre>
 *
 * where <code>XXXProvider</code> is an implementation of this interface.
 *
 * A custom buffered image provider is also registered as drawer {@link LLUIDisplay#registerUIDrawer(UIDrawing)} and
 * custom image drawer {@link LLUIDisplay#registerUIImageDrawer(UIImageDrawing)}.
 * <p>
 * So it is used to:
 * <ul>
 * <li>create image buffers (with {@link #newBufferedImage(int, int)},</li>
 * <li>draw in these buffer (with methods from {@link UIDrawing}, for example
 * {@link #drawLine(MicroUIGraphicsContext, int, int, int, int)}),</li>
 * <li>draw these buffers in other buffers - image or display (with methods from {@link UIImageDrawing}, for example
 * {@link #draw(MicroUIGraphicsContext, ej.microui.display.MicroUIImage, int, int, int, int, int, int, int)}),</li>
 * <li>decode images from the image generator (with {@link #decode(byte[], int, int, int, boolean)}).</li>
 * </ul>
 */
public interface BufferedImageProvider extends UIDrawing, UIImageDrawing {

	/**
	 * Creates a new buffered image.
	 * <p>
	 * Note that a buffered image is an image which is not managed by the Graphics Engine' software algorithms. The
	 * Platform has the responsibility to check if the drawing target is a custom image or not.
	 * <p>
	 * The Platform object can be retrieved at runtime by calling {@link Image#getRAWImage()}:
	 *
	 * <pre>
	 * public void fillRectangle(MicroUIGraphicsContext gc, int x1, int y1, int x2, int y2) {
	 * 	MyCustomImage customImage = (MyCustomImage) gc.getImage().getRAWImage();
	 * 	// draw in a custom buffered image
	 * 	// ...
	 * }
	 * </pre>
	 *
	 * For the custom drawings, the {@link MicroUIGraphicsContext} (that targets a custom buffered image) can be used as
	 * a standard graphics context (see {@link LLUIPainter}).
	 *
	 * @param width
	 *            the buffered image width
	 * @param height
	 *            the buffered image height
	 *
	 * @return an object that represents the buffered image
	 */
	Object newBufferedImage(int width, int height);

}
