/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import org.junit.Assume;

import ej.bon.Constants;
import ej.microui.MicroUI;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microui.display.ResourceImage;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Provides utility methods for tests.
 */
@SuppressWarnings("nls")
public class TestUtilities {

	/**
	 * Boolean constant (see BON {@link Constants}) to enable the debug. Default value is <code>false</code>.
	 */
	public static final String DEBUG_CONSTANT = "com.microej.microvg.test.debug";

	/**
	 * Boolean constant (see BON {@link Constants}) to disable gradient tests.
	 */
	public static final String SKIP_GRADIENT_TESTS_CONSTANT = "tests.gradients.skip";

	private static final int MAX_COLOR_COMPONENT_VALUE = 0xff;
	private static final int ALPHA_SHIFT = 24;
	private static final int RED_SHIFT = 16;
	private static final int GREEN_SHIFT = 8;
	private static final int BLUE_SHIFT = 0;
	private static final int ALPHA_MASK = MAX_COLOR_COMPONENT_VALUE << ALPHA_SHIFT;
	private static final int RED_MASK = MAX_COLOR_COMPONENT_VALUE << RED_SHIFT;
	private static final int GREEN_MASK = MAX_COLOR_COMPONENT_VALUE << GREEN_SHIFT;
	private static final int BLUE_MASK = MAX_COLOR_COMPONENT_VALUE << BLUE_SHIFT;
	private static final int MAX_OPAQUE_COLOR_VALUE = 0xffffff;

	/**
	 * Color used as background color before each test.
	 *
	 * @see TestUtilities#clearScreen()
	 */
	public static final int BACKGROUND_COLOR = Colors.BLACK;

	/**
	 * Enumeration that identifies the current target.
	 */
	public enum VEEPort {
		/**
		 * Current target is the embedded device
		 */
		Embedded,

		/**
		 * Current target is the simulator (backend AWT)
		 */
		Simulator,

		/**
		 * Current target is the ACK (Android kit)
		 */
		Android,

		/**
		 * Current target is the ICK (iOS kit) or MacOS or Windows over JavaFX.
		 */
		JavaFX;
	}

	private static VEEPort veePortType;

	private TestUtilities() {
		// Prevent instantiation.
	}

	/**
	 * Checks that the pixel at given (x,y) coordinates has the expected color.
	 *
	 * <p>
	 * This method compares two colors component-by-component (R,G,B). A tolerance of 8 is used when comparing
	 * respective components (see {@link #compare(int, int)}).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x-coordinate of the pixel to test.
	 * @param y
	 *            the y-coordinate of the pixel to test.
	 * @param g
	 *            the graphics context.
	 * @param expectedColor
	 *            the color to test the display color against.
	 */
	public static void check(String label, int x, int y, GraphicsContext g, int expectedColor) {
		int expectedDisplayColor = Display.getDisplay().getDisplayColor(expectedColor);
		int readColor = g.readPixel(x, y);
		if (!compare(expectedDisplayColor, readColor)) {
			System.out.println("TestUtilities.check() at " + x + "," + y + ": expected = 0x"
					+ Integer.toHexString(expectedDisplayColor) + ", actual = 0x" + Integer.toHexString(readColor));
		}

		if (Constants.getBoolean(DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, x, y, 3, 3);
			Display.getDisplay().flush();
		}
		assertTrue(label, compare(expectedDisplayColor, readColor));
	}

	/**
	 * Checks that the pixel at given (x,y) coordinates has not the specified color.
	 *
	 * <p>
	 * This method compares two colors component-by-component (R,G,B). A tolerance of 8 is used when comparing
	 * respective components (see {@link #compare(int, int)}).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x-coordinate of the pixel to test.
	 * @param y
	 *            the y-coordinate of the pixel to test.
	 * @param g
	 *            the graphics context.
	 * @param expectedColor
	 *            the color to test the display color against.
	 */
	public static void checkNot(String label, int x, int y, GraphicsContext g, int expectedColor) {
		int displayColor = Display.getDisplay().getDisplayColor(expectedColor);
		int readColor = g.readPixel(x, y);
		assertFalse(label, compare(displayColor, readColor));
	}

	/**
	 * Checks that all the pixels in the specified area have the expected color.
	 *
	 * <p>
	 * The padding is the amount of space inside the specified area (along the border) to be excluded from the test.
	 * This can be convenient to avoid anti-aliasing issues on the edges when testing pixels.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 * @param padding
	 *            the space inside the area, along the border of this area, to be excluded from the test.
	 */
	public static void checkArea(String label, int color, int x, int y, int width, int height, int padding) {
		checkArea(label, color, x + padding, y + padding, width - 2 * padding, height - 2 * padding);
	}

	/**
	 * Checks that all the pixels in the specified area have the expected color.
	 *
	 * <p>
	 * This method uses a padding of 0, that means that no pixels are excluded from the test in the specified area.
	 * Equivalent to calling {@link #checkArea(String, int, int, int, int, int, int)} with padding equals 0.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 */
	public static void checkArea(String label, int color, int x, int y, int width, int height) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		int expectedColor = display.getDisplayColor(color);
		boolean result = true;
		int translationX = g.getTranslationX();
		int translationY = g.getTranslationY();

		for (int i = 0; i < width; i++) {
			int xi = x + i;
			if ((xi + translationX < 0) || (xi + translationX >= display.getWidth())) {
				continue;
			}
			for (int j = 0; j < height; j++) {
				int yj = y + j;
				if ((yj + translationY < 0) || (yj + translationY >= display.getHeight())) {
					continue;
				}

				int readColor = g.readPixel(xi, yj) & MAX_OPAQUE_COLOR_VALUE;
				boolean compare = compare(expectedColor, readColor);
				if (!compare && Constants.getBoolean(DEBUG_CONSTANT)) {
					System.out.println("TestUtilities.checkArea() 0x" + Integer.toHexString(expectedColor) + " 0x"
							+ Integer.toHexString(readColor) + " x=" + xi + " y=" + yj);
				}
				result &= compare;
			}
		}

		if (Constants.getBoolean(DEBUG_CONSTANT)) {
			g.setColor(Colors.RED);
			Painter.drawRectangle(g, x, y, width, height);
			display.flush();
		}

		assertTrue(label, result);
	}

	/**
	 * Checks the expected color in the surrounding periphery of the specified area.
	 *
	 * <p>
	 * The periphery is made of 4 rectangular areas located on the edges of the target area: top, bottom, left and
	 * right. The specified thickness defines the size of these areas.
	 *
	 * <p>
	 * The padding is the amount of space inside the test areas (along the border) to be excluded from the test. This
	 * can be convenient to avoid anti-aliasing issues on the edges when testing pixels.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color in the peripheral area
	 * @param x
	 *            the x origin of the target area (top left corner)
	 * @param y
	 *            the y origin of the target area (top left corner)
	 * @param width
	 *            the width of the target area
	 * @param height
	 *            the height of the target area
	 * @param thickness
	 *            the thickness of the peripheral area (i.e., test area)
	 * @param padding
	 *            the padding of the peripheral area (i.e., test area)
	 * @param assertOnError
	 *            true to throw an assert on error, false to let the caller manage the error by reading the returned
	 *            value
	 * @return the number of errors
	 */
	public static int checkPeripheralArea(String label, int color, int x, int y, int width, int height, int thickness,
			int padding, boolean assertOnError) {
		int errors = 0;
		try {
			checkArea(label + " (outside top)", color, x, y - thickness, width, thickness, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside bottom)", color, x, y + height, width, thickness, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside left)", color, x - thickness, y, thickness, height, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}
		try {
			checkArea(label + " (outside right)", color, x + width, y, thickness, height, padding);
		} catch (AssertionError e) {
			System.out.println(e);
			errors++;
		}

		if (assertOnError) {
			assertEquals(0, errors);
		}

		return errors;
	}

	/**
	 * Checks the expected color in the surrounding periphery of the specified area.
	 * <p>
	 * Equivalent to
	 *
	 * <pre>
	 * checkPeripheralArea(label, color, x, y, width, height, thickness, padding, true);
	 * </pre>
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param color
	 *            the expected color in the peripheral area
	 * @param x
	 *            the x origin of the target area (top left corner)
	 * @param y
	 *            the y origin of the target area (top left corner)
	 * @param width
	 *            the width of the target area
	 * @param height
	 *            the height of the target area
	 * @param thickness
	 *            the thickness of the peripheral area (i.e., test area)
	 * @param padding
	 *            the padding of the peripheral area (i.e., test area)
	 * @see TestUtilities#checkPeripheralArea(String, int, int, int, int, int, int, int, boolean)
	 */
	public static void checkPeripheralArea(String label, int color, int x, int y, int width, int height, int thickness,
			int padding) {
		checkPeripheralArea(label, color, x, y, width, height, thickness, padding, true);
	}

	/**
	 * Compares two colors.
	 *
	 * @param displayColor
	 *            the reference
	 * @param readColor
	 *            the color to check
	 * @return true if the difference on each component of the colors are lower than 0x08.
	 */
	public static boolean compare(int displayColor, int readColor) {
		return compareComponent(getRed(displayColor), getRed(readColor))
				&& compareComponent(getGreen(displayColor), getGreen(readColor))
				&& compareComponent(getBlue(displayColor), getBlue(readColor));
	}

	private static boolean compareComponent(int displayComponent, int readComponent) {
		return compareComponent(readComponent, displayComponent, 0x08);
	}

	/**
	 * Compare two colors components.
	 *
	 * @param component1
	 *            the component of the first color
	 * @param component2
	 *            the component of the second color
	 * @param componentTolerance
	 *            the acceptable tolerance
	 * @return true if the difference between the two components is smaller or equal to the tolerance.
	 */
	public static boolean compareComponent(int component1, int component2, int componentTolerance) {
		int diff = Math.abs(component1 - component2);
		return diff <= componentTolerance;
	}

	/**
	 * Gets a color opacity (alpha) component.
	 *
	 * @param color
	 *            the color.
	 * @return the opacity component.
	 */
	public static int getAlpha(int color) {
		return (color & ALPHA_MASK) >>> ALPHA_SHIFT;
	}

	/**
	 * Gets a color red component.
	 *
	 * @param color
	 *            the color.
	 * @return the red component.
	 */
	public static int getRed(int color) {
		return (color & RED_MASK) >>> RED_SHIFT;
	}

	/**
	 * Gets a color green component.
	 *
	 * @param color
	 *            the color.
	 * @return the green component.
	 */
	public static int getGreen(int color) {
		return (color & GREEN_MASK) >>> GREEN_SHIFT;
	}

	/**
	 * Gets a color blue component.
	 *
	 * @param color
	 *            the color.
	 * @return the blue component.
	 */
	public static int getBlue(int color) {
		return (color & BLUE_MASK) >>> BLUE_SHIFT;
	}

	/**
	 * Gets a color with specified alpha applied.
	 *
	 * @param color
	 *            the color.
	 * @param alpha
	 *            the alpha, between {@link GraphicsContext#TRANSPARENT} and {@link GraphicsContext#OPAQUE}.
	 * @return the color with alpha.
	 */
	public static int getColorWithAlpha(int color, int alpha) {
		return (color & MAX_OPAQUE_COLOR_VALUE) | (alpha << ALPHA_SHIFT);
	}

	/**
	 * Gets a color with the four components.
	 *
	 * @param a
	 *            the opacity
	 * @param r
	 *            the red component
	 * @param g
	 *            the green component
	 * @param b
	 *            the blue component
	 * @return the 32-bit color
	 */
	public static int getColor(int a, int r, int g, int b) {
		return ((a & MAX_COLOR_COMPONENT_VALUE) << TestUtilities.ALPHA_SHIFT)
				| ((r & MAX_COLOR_COMPONENT_VALUE) << TestUtilities.RED_SHIFT)
				| ((g & MAX_COLOR_COMPONENT_VALUE) << TestUtilities.GREEN_SHIFT)
				| ((b & MAX_COLOR_COMPONENT_VALUE) << TestUtilities.BLUE_SHIFT);
	}

	/**
	 * Blends two colors with the given opacity
	 *
	 * @param color
	 *            the foreground color
	 * @param background
	 *            the background color
	 * @param alpha
	 *            the opacity to apply
	 * @return the blended color
	 */
	public static int blendColors(int color, int background, int alpha) {
		int red = (getRed(color) * alpha + getRed(background) * (MAX_COLOR_COMPONENT_VALUE - alpha))
				/ MAX_COLOR_COMPONENT_VALUE;
		int green = (getGreen(color) * alpha + getGreen(background) * (MAX_COLOR_COMPONENT_VALUE - alpha))
				/ MAX_COLOR_COMPONENT_VALUE;
		int blue = (getBlue(color) * alpha + getBlue(background) * (MAX_COLOR_COMPONENT_VALUE - alpha))
				/ MAX_COLOR_COMPONENT_VALUE;

		return (red << RED_SHIFT) | (green << GREEN_SHIFT) | blue;
	}

	/**
	 * Clears the screen with black.
	 */
	public static void clearScreen() {
		clearScreen(Colors.BLACK);
	}

	/**
	 * Clears the screen with the specified color.
	 *
	 * @param color
	 *            the color to apply
	 */
	public static void clearScreen(int color) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.setColor(color);
		Painter.fillRectangle(g, -g.getTranslationX(), -g.getTranslationY(), display.getWidth(), display.getHeight());
	}

	/**
	 * Loads the dedicated testsuite font
	 *
	 * @return a vector font over firstfont.ttf.
	 */
	public static VectorFont getTestFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	/**
	 * Debug function to draw a grid when drawing a text on a circle
	 *
	 * @param font
	 *            the font used to draw the text
	 * @param fontSize
	 *            the font size
	 * @param radius
	 *            the circle radius in degrees
	 * @param x
	 *            the destination X position
	 * @param y
	 *            the destination Y postion
	 * @param direction
	 *            the text direction
	 */
	public static void drawStringOnCircleGrid(VectorFont font, float fontSize, float radius, int x, int y,
			Direction direction) {

		// Clockwise direction
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		float bottomCircleRadius = topCircleRadius - font.getHeight(fontSize);

		// If counter-Clockwise direction
		if (direction == Direction.COUNTER_CLOCKWISE) {
			topCircleRadius = radius - font.getBaselinePosition(fontSize);
			bottomCircleRadius = topCircleRadius + fontSize;
		}

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.setColor(Colors.GREEN);
		Painter.drawCircle(g, (int) (x - topCircleRadius), (int) (y - topCircleRadius), (int) (topCircleRadius * 2));

		g.setColor(Colors.LIME);
		Painter.drawCircle(g, (int) (x - bottomCircleRadius), (int) (y - bottomCircleRadius),
				(int) (bottomCircleRadius * 2));

		g.setColor(Colors.YELLOW);
		Painter.drawCircle(g, (int) (x - radius), (int) (y - radius), (int) (radius * 2));

		g.setColor(Colors.RED);
		Painter.drawLine(g, x, y - 10, x, y + 10);
		Painter.drawLine(g, x - 10, y, x + 10, y);

		display.flush();
	}

	/**
	 * Return whether the testsuite is running on simulator.
	 *
	 * @return whether the testsuite is running on simulator.
	 *
	 * @deprecated call {@link #isOn(VEEPort)} instead
	 */
	@Deprecated
	static public boolean isOnSimulator() {
		return isOn(VEEPort.Simulator);
	}

	/**
	 * Return whether the testsuite is running on Android.
	 *
	 * @return whether the testsuite is running on Android.
	 *
	 * @deprecated call {@link #isOn(VEEPort)} instead
	 */
	@Deprecated
	public static boolean isOnAndroid() {
		return isOn(VEEPort.Android);
	}

	/**
	 * Return whether the testsuite is running on the given VEE Port.
	 *
	 * @param type
	 *            the VEE Port type to test
	 *
	 * @return whether the testsuite is running on the given VEE Port.
	 */
	public static boolean isOn(VEEPort type) {
		return getVEEPortType() == type;
	}

	/**
	 * Measures character bbox width
	 *
	 * A character box is splitted in 3 parts a|b|c
	 *
	 * <li>a is the padding space between the start of the bbox and the start of the glyph
	 * <li>b is the glyph
	 * <li>c is the padding space between the end of the glyph and the end of the bbox
	 *
	 * font.measureStringWidth of a 1 character string returns "b". If the string as 2 identical characters it returns:
	 * "a + b + b + c = a + 2b + c"
	 *
	 * Thus the bbox width is equal to (measureStringWidth(2*character) - measureStringWidth(character)
	 *
	 * @param s
	 *            the string to measure
	 * @param font
	 *            the font to use
	 * @param fontSize
	 *            the font size
	 * @return the string width in pixels
	 */
	public static float measureCharWidth(String s, VectorFont font, float fontSize) {
		return (font.measureStringWidth(s + s, fontSize) - font.measureStringWidth(s, fontSize));
	}

	/**
	 * Compares the drawings made on the display with an image.
	 *
	 * @param filename
	 *            the image path
	 * @param x
	 *            the X position of drawing bounding box
	 * @param y
	 *            the Y position of drawing bounding box
	 * @param width
	 *            the width of drawing bounding box
	 * @param height
	 *            the height of drawing bounding box
	 * @param maxDiff
	 *            the maximum number of allowed differences before throwing an assert
	 */
	public static void compareDisplay(String filename, int x, int y, int width, int height, float maxDiff) {
		GraphicsContext g = Display.getDisplay().getGraphicsContext();

		// "load" the image instead of "getting" it: more portable according to the VEE Port
		String imagePath = "/images/" + filename + ".png";
		try (ResourceImage ref = ResourceImage.loadImage(imagePath)) {

			int widthRef = ref.getWidth();
			int heightRef = ref.getHeight();

			if (widthRef != width || heightRef != height) {
				System.err.println("Incorrect screen size: screenshot = " + widthRef + "x" + heightRef + ", screen = "
						+ width + "x" + height);
			}

			int incorrectPixels = 0;
			int nbPixel = 0;
			for (int i = 0; i < widthRef; i++) {
				for (int j = 0; j < heightRef; j++) {
					int k = x + i;
					int l = y + j;

					int colorRef = ref.readPixel(i, j);

					int colorDisp = g.readPixel(k, l);

					if (((colorRef & MAX_OPAQUE_COLOR_VALUE) == BACKGROUND_COLOR)
							&& ((colorDisp & MAX_OPAQUE_COLOR_VALUE) == BACKGROUND_COLOR)) {
						continue;
					}

					int componentRef = getMaxComponent(colorRef);
					int componentDisp = getMaxComponent(colorDisp);

					int pixelDiff = Math.abs(componentRef - componentDisp);
					incorrectPixels += pixelDiff;
					nbPixel += 1;
				}
			}

			double percentIncorrectPixels = (double) incorrectPixels / (nbPixel * 255);

			String label = "compareDisplay_" + filename + " - percentIncorrectPixels=" + percentIncorrectPixels;
			System.out.println(label);
			assertFalse(label, percentIncorrectPixels > maxDiff);
		}
	}

	private static int getMaxComponent(int color) {
		return Math.max(getRed(color), Math.max(getGreen(color), getBlue(color)));
	}

	/**
	 * Compares both matrices and throws an error when different or equal.
	 *
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 * @param errorIfDifferent
	 *            true to throw an error when different, false to throw an error when equal.
	 */
	public static void checkMatrix(Matrix a, Matrix b, boolean errorIfDifferent) {
		float[] ma = a.getSNIContext();
		float[] mb = b.getSNIContext();
		checkMatrix(ma, mb, errorIfDifferent);
	}

	/**
	 * Compares both matrices and throws an error when different or equal. Both matrices must have the same size.
	 *
	 * @param a
	 *            the first matrix
	 * @param b
	 *            the second matrix
	 * @param errorIfDifferent
	 *            true to throw an error when different, false to throw an error when equal.
	 */
	public static void checkMatrix(float[] a, float[] b, boolean errorIfDifferent) {

		assertEquals("matrices sizes", a.length, b.length);

		boolean same = true;
		for (int i = 0; i < a.length; i++) {
			if (a[i] != b[i]) {
				same = false;
			}
		}

		if (errorIfDifferent) {
			assertTrue("matrices compare", same);
		} else {
			assertFalse("matrices are identical", same);
		}
	}

	/**
	 * Utility method to start or not MicroUI according to the target.
	 * <p>
	 * When MicroUI is not started again, the display's graphics context is reseted.
	 */
	public static void startMicroUI() {
		if (!MicroUI.isStarted() || isOn(VEEPort.Embedded)) {
			MicroUI.start();
		} else {
			// Reset graphics context
			GraphicsContext g = Display.getDisplay().getGraphicsContext();
			g.reset();
		}
	}

	/**
	 * Utility method to stop or not MicroUI according to the target.
	 */
	static public void stopMicroUI() {
		if (isOn(VEEPort.Embedded) || isOn(VEEPort.Simulator)) {
			MicroUI.stop();
		}
	}

	/**
	 * Compares all the pixels in the specified areas. Both areas must fit the display area (with or without a
	 * translation).
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param ex
	 *            the x origin of the expected area (top-left)
	 * @param ey
	 *            the y origin of the expected area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 */
	public static void compareAreas(String label, int x, int y, int ex, int ey, int width, int height) {
		compareAreas(label, x, y, ex, ey, width, height, 0);
	}

	/**
	 * Compares all the pixels in the specified areas. Both areas must fit the display area (with or without a
	 * translation). A tolerance (a percentage less than 10%) is allowed.
	 *
	 * @param label
	 *            the identifying message for the AssertionError (null okay)
	 * @param x
	 *            the x origin of the area (top-left)
	 * @param y
	 *            the y origin of the area (top-left)
	 * @param ex
	 *            the x origin of the expected area (top-left)
	 * @param ey
	 *            the y origin of the expected area (top-left)
	 * @param width
	 *            the width of the area
	 * @param height
	 *            the height of the area
	 * @param tolerance
	 *            maximum number of pixels (in percantage) that can be different (max 0.1 for 10%)
	 */
	public static void compareAreas(String label, int x, int y, int ex, int ey, int width, int height,
			float tolerance) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		boolean result = true;
		int counter = 0;

		try {

			for (int i = 0; i < width; i++) {
				for (int j = 0; j < height; j++) {
					int readColor = g.readPixel(x + i, y + j);
					int expectedColor = g.readPixel(ex + i, ey + j);
					boolean compare = compare(expectedColor, readColor);
					if (!compare) {
						++counter;
						if (Constants.getBoolean(DEBUG_CONSTANT)) {
							System.out.print("TestUtilities.compareArea() at ");
							System.out.print(i);
							System.out.print(",");
							System.out.print(j);
							System.out.println(" 0x" + Integer.toHexString(expectedColor) + " 0x"
									+ Integer.toHexString(readColor));
						}
					}
					result &= compare;
				}
			}
		} catch (Exception e) {
			// out of bounds
			result = false;
		}

		float tr = ((float) counter) / (width * height);
		if (tr > 0) {
			System.out.println("Invalid pixels ratio: " + tr);
			assertTrue(label + " tolerance", tr < tolerance);
		}

		if (tolerance > 0) {
			assertTrue(label + " tolerance too high (higher than 0.1)", tolerance <= 0.1f); // 10%
		} else {
			assertTrue(label, result);
		}
	}

	private static VEEPort getVEEPortType() {
		if (null == veePortType) {
			switch (Constants.getInt("com.microej.microvg.test.veeport")) {
			case 1:
				veePortType = VEEPort.Android;
				break;
			case 2:
				veePortType = VEEPort.JavaFX;
				break;
			default:
				// backward compatible (deprecated)
				if (Constants.getBoolean("com.microej.microvg.test.android")) {
					veePortType = VEEPort.Android;
				} else if (Constants.getBoolean("com.microej.library.microui.onS3")) {
					veePortType = VEEPort.Simulator;
				} else {
					veePortType = VEEPort.Embedded;
				}
				break;
			}
		}
		return veePortType;
	}

	/**
	 * Prevents the test from running if gradient tests are disabled.
	 *
	 * @see #SKIP_GRADIENT_TESTS_CONSTANT
	 */
	public static void assumeGradients() {
		Assume.assumeFalse(Boolean.getBoolean(SKIP_GRADIENT_TESTS_CONSTANT));
	}
}
