/*
 * Java
 *
 * Copyright 2021-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorImage;

/**
 * Tests {@link VectorImage#getImage(String)} method.
 */
@SuppressWarnings("nls")
public class TestGetImage {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Tests getting an existing image.
	 */
	@Test
	public static void testExistingImage() {
		VectorImage.getImage("/com/microej/microvg/test/mascot.xml");
		Assert.assertTrue("get image", true);
	}

	/**
	 * Test that trying to load an image with an empty path throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testInvalidVoidPathException() {
		VectorImage.getImage("");
	}

	/**
	 * Test that trying to load an image with an empty path results in the correct error code.
	 */
	@Test
	public void testInvalidVoidPathErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorImage.getImage("");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("void path error code", errorCode, VectorGraphicsException.IMAGE_INVALID_PATH);
	}

	/**
	 * Test that trying to load an image with an empty path throws the correct exception. A path is invalid if it
	 * doesn't start with a path separator.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testInvalidEmptyPathException() {
		VectorImage.getImage(" ");
	}

	/**
	 * Test that trying to load an image with an empty path results in the correct error code. A path is invalid if it
	 * doesn't start with a path separator.
	 */
	@Test
	public void testInvalidEmptyPathErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorImage.getImage(" ");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("invalid empty path error code", errorCode, VectorGraphicsException.IMAGE_INVALID_PATH);
	}

	/**
	 * Test that trying to load an image with an empty path throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testValidEmptyPathException() {
		VectorImage.getImage("/");
	}

	/**
	 * Test that trying to load an image with an empty path results in the correct error code.
	 */
	@Test
	public void testValidEmptyPathErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorImage.getImage("/");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("valid empty path error code", errorCode, VectorGraphicsException.IMAGE_INVALID_PATH);
	}

	/**
	 * Tests loading an existing image without leading slash throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testMissingLeadingSlashException() {
		VectorImage.getImage("com/microej/microvg/test/mascot.xml");
	}

	/**
	 * Tests loading an existing image without leading slash results in the correct error code.
	 */
	@Test
	public void testMissingLeadingSlashErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorImage.getImage("com/microej/microvg/test/mascot.xml");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("missing leading slash error code", errorCode, VectorGraphicsException.IMAGE_INVALID_PATH);
	}

	/**
	 * Test that loading an image that does not exist in the classpath throws the correct exception.
	 */
	@Test(expected = VectorGraphicsException.class)
	public void testNonExistingFileException() {
		VectorImage.getImage("/images/themascot.xml");
	}

	/**
	 * Test that loading an image that does not exist in the classpath results in the correct error code.
	 */
	@Test
	public void testNonExistingFileErrorCode() {
		int errorCode = Integer.MAX_VALUE;

		try {
			VectorImage.getImage("/images/themascot.xml");
		} catch (VectorGraphicsException e) {
			errorCode = e.getErrorCode();
		}

		Assert.assertEquals("non existing file error code", errorCode, VectorGraphicsException.IMAGE_INVALID_PATH);
	}

}
