/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.cascading;

import ej.basictool.ArrayTools;
import ej.mwt.Widget;
import ej.mwt.style.EditableStyle;
import ej.mwt.style.Style;
import ej.mwt.stylesheet.Stylesheet;
import ej.mwt.stylesheet.selector.Selector;

/**
 * Cascading stylesheet implementation strongly inspired by CSS.
 * <p>
 * This stylesheet contains:
 * <ul>
 * <li>a default style that defines all the attributes,</li>
 * <li>a set of rules with a selector and a partial style.</li>
 * </ul>
 * <p>
 * The style of a widget is determined following these steps:
 * <ol>
 * <li>create an empty result style (no attributes set),</li>
 * <li>merge widget with matching selectors rules (set by {@link #getSelectorStyle(Selector)}),</li>
 * <li>recursively merge with inherited attributes of parents (repeat previous step for parent recursively) (see
 * {@link Widget#getParent()}),</li>
 * <li>merge global style (set by {@link #getDefaultStyle()}).</li>
 * </ol>
 * The merge consists in completing the result style with all the set attributes of another style (see {@link Style}).
 * The result style is complete at the end of the resolution.
 */
public class CascadingStylesheet implements Stylesheet {

	/**
	 * Default style.
	 */
	private EditableStyle defaultStyle;
	/**
	 * Rules.
	 */
	// The list ensures the definition order.
	private Object[] selectorStyles; // contains each rule sequentially (selector and style)

	/**
	 * Creates a new cascading stylesheet.
	 */
	public CascadingStylesheet() {
		// Equivalent of #reset()
		this.defaultStyle = new EditableStyle();
		this.selectorStyles = new Object[0];
	}

	@Override
	public Style getStyle(Widget widget) {
		CascadingStyle resultingStyle = new CascadingStyle(this.defaultStyle);

		// Global style selectors only.
		mergeSelectors(widget, resultingStyle);

		// Merge with parents inherited attributes.
		Widget parentWidget = widget.getParent();
		if (parentWidget != null) {
			Style parentStyle = parentWidget.getStyle();
			resultingStyle.inheritMerge(parentStyle);
		}

		resultingStyle.updateHashCode();

		return resultingStyle;
	}

	private void mergeSelectors(Widget widget, CascadingStyle resultingStyle) {
		Object[] selectorStyles = this.selectorStyles;

		for (int i = 0; i < selectorStyles.length; i += 2) {
			if (((Selector) selectorStyles[i]).appliesToWidget(widget)) {
				CascadingStyle style = (CascadingStyle) selectorStyles[i + 1];
				assert (style != null);
				resultingStyle.merge(style);
			}
		}
	}

	/**
	 * Gets the default style. The style can be modified.
	 * <p>
	 * This style is used as the root style of the cascading resolution. Its initial attributes are equal to the values
	 * defined in {@link ej.mwt.style.DefaultStyle}.
	 *
	 * @return the editable default style.
	 */
	public EditableStyle getDefaultStyle() {
		return this.defaultStyle;
	}

	/**
	 * Resets the default style attributes to their initial value.
	 */
	public void resetDefaultStyle() {
		this.defaultStyle = new EditableStyle();
	}

	/**
	 * Gets the style for a selector. The style can be modified.
	 * <p>
	 * This style is applied to the widgets matching the selector.
	 *
	 * @param selector
	 *            the selector.
	 * @return the editable style for the given selector.
	 */
	public EditableStyle getSelectorStyle(Selector selector) {
		// Get the array in local to avoid synchronization (with add and remove).
		Object[] selectorStyles = this.selectorStyles;
		int selectorIndex = getSelectorIndex(selector, selectorStyles);
		if (selectorIndex != -1) {
			CascadingStyle style = (CascadingStyle) selectorStyles[selectorIndex + 1];
			assert (style != null);
			return style;
		} else {
			CascadingStyle style = new CascadingStyle();
			int index = getIndex(selector, selectorStyles);
			// Add the selector at the first place to ensure order (last added is resolved first).
			selectorStyles = ArrayTools.grow(selectorStyles, index, 2);
			selectorStyles[index] = selector;
			selectorStyles[index + 1] = style;
			this.selectorStyles = selectorStyles;
			return style;
		}
	}

	/**
	 * Resets the style attributes for a selector.
	 *
	 * @param selector
	 *            the selector.
	 */
	public void resetSelectorStyle(Selector selector) {
		// Get the array in local to avoid synchronization (with add and remove).
		Object[] selectorStyles = this.selectorStyles;
		int selectorIndex = getSelectorIndex(selector, selectorStyles);
		if (selectorIndex != -1) {
			this.selectorStyles = ArrayTools.shrink(selectorStyles, selectorIndex, 2);
		}
	}

	/**
	 * Resets the stylesheet to its initial state.
	 */
	public void reset() {
		// Duplicated in the constructor.
		this.defaultStyle = new EditableStyle();
		this.selectorStyles = new Object[0];
	}

	private static int getIndex(Selector selector, Object[] selectorStyles) {
		int specificity = selector.getSpecificity();
		int minIndex = 0;
		int maxIndex = selectorStyles.length / 2;
		while (maxIndex - minIndex > 1) {
			int currentIndex = (maxIndex - minIndex) / 2 + minIndex;
			if (isMoreSpecific(selectorStyles, currentIndex * 2, specificity)) {
				minIndex = currentIndex;
			} else {
				maxIndex = currentIndex;
			}
		}
		if (maxIndex > minIndex) {
			if (isMoreSpecific(selectorStyles, minIndex * 2, specificity)) {
				return maxIndex * 2;
			} else {
				return minIndex * 2;
			}
		}
		return minIndex * 2;
	}

	private static boolean isMoreSpecific(Object[] selectorStyles, int index, int specificity) {
		int specificityCandidate = ((Selector) selectorStyles[index]).getSpecificity();
		// If equals, the first added remains first.
		return specificityCandidate > specificity;
	}

	private static int getSelectorIndex(Selector selector, Object[] selectorStyles) {
		for (int i = 0; i < selectorStyles.length; i += 2) {
			if (selector.equals(selectorStyles[i])) {
				return i;
			}
		}
		return -1;
	}
}
