/*
 * Java
 *
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.property;

import ej.annotation.NonNull;
import ej.annotation.Nullable;
import ej.basictool.map.PackedMap;
import ej.kf.Feature;
import ej.kf.Feature.State;
import ej.kf.FeatureStateListener;
import ej.kf.Kernel;
import ej.kf.Module;
import ej.property.PropertyPermission;
import ej.property.SharedPropertyRegistry;
import ej.property.registry.SimplePropertyRegistry;


/**
 * Implementation of shared property registry over Kernel and Features.
 */
public class SharedPropertyRegistryKF extends SimplePropertyRegistry
implements SharedPropertyRegistry, FeatureStateListener {

	/**
	 * Map which contains shared properties registered by features.
	 *
	 * Methods modifying these maps are synchronized using the monitor of the map.
	 */
	private final PackedMap<Feature, PackedMap<String, String>> sharedPropertiesByFeatures;

	/**
	 * Creates a new shared registry KF.
	 */
	public SharedPropertyRegistryKF() {
		this.sharedPropertiesByFeatures = new PackedMap<>();
	}

	@Override
	@Nullable
	public String setProperty(String key, String value) {
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (contextOwner == Kernel.getInstance()) {
			// Property owned by the Kernel.
			return super.setProperty(key, value);
		} else {
			// Property owned by a Feature.
			Kernel.enter();

			checkPropertyPermission(key, PropertyPermission.SET_ACTION);

			PackedMap<String, String> featureProperties;
			PackedMap<Feature, PackedMap<String, String>> sharedPropertiesByFeatures = this.sharedPropertiesByFeatures;

			synchronized (sharedPropertiesByFeatures) {
				if (sharedPropertiesByFeatures.isEmpty()) {
					Kernel.addFeatureStateListener(this);
				}

				Feature featureOwner = (Feature) contextOwner;
				featureProperties = sharedPropertiesByFeatures.get(featureOwner);
				if (featureProperties == null) {
					// This is the first time this Feature sets a property.
					featureProperties = new PackedMap<>();
					sharedPropertiesByFeatures.put(featureOwner, featureProperties);
				}
			}

			synchronized (featureProperties) {
				return featureProperties.put(key, value);
			}
		}
	}

	@Override
	@Nullable
	public String removeProperty(String key) {
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (contextOwner == Kernel.getInstance()) {
			// Property owned by the Kernel.
			return super.removeProperty(key);
		} else {
			// Property owned by a Feature.
			Kernel.enter();

			checkPropertyPermission(key, PropertyPermission.REMOVE_ACTION);

			PackedMap<Feature, PackedMap<String, String>> sharedPropertiesByFeatures = this.sharedPropertiesByFeatures;

			PackedMap<String, String> sharedProperties = sharedPropertiesByFeatures.get(contextOwner);
			if (sharedProperties != null) {
				synchronized (sharedProperties) {
					String result = sharedProperties.remove(key);
					if (result != null) {
						if (sharedProperties.isEmpty()) {
							// This Feature no longer register some properties.
							synchronized (sharedPropertiesByFeatures) {
								sharedPropertiesByFeatures.remove(contextOwner);
								if (sharedPropertiesByFeatures.isEmpty()) {
									Kernel.removeFeatureStateListener(this);
								}
							}
						}
						return result;
					}
				}
			}
			// Not found.
			return null;
		}
	}

	/**
	 * Searches for the property with the specified key in the property lists. If the property is not found in any map
	 * of properties, then the method returns null.
	 *
	 * Different property maps of properties exist. The order in which they are searched in is:
	 * <ol>
	 * <li>Current application,</li>
	 * <li>Kernel,</li>
	 * <li>Trusted applications,</li>
	 * <li>Other applications.</li>
	 * </ol>
	 *
	 * In the case a property value is set to null, then it is considered as non-existent in its map and the search goes
	 * on, possibly returning a shared-property sharing the same key.
	 *
	 * @param key
	 *            the property key.
	 * @return the value of the property list with the specified key value.
	 */
	@Override
	@Nullable
	public String getProperty(String key) {
		checkPropertyPermission(key, PropertyPermission.GET_ACTION);

		// Search the property in the ones owned by the current context owner.
		@SuppressWarnings("null")
		@NonNull
		Module contextOwner = Kernel.getContextOwner();
		if (!Kernel.isInKernelMode()) {
			PackedMap<String, String> sharedProperties = this.sharedPropertiesByFeatures.get(contextOwner);

			if (sharedProperties != null) {
				String value = sharedProperties.get(key);
				if (value != null) {
					return value;
				}
			}
		}

		// Search the property in the ones provided by the kernel.
		String value = super.getProperty(key);
		if (value != null) {
			return value;
		}

		Kernel.enter();

		// Create the same string owned by the kernel in order to call equals().
		String keyKernel = new String(key);
		// Search the service in the ones provided by features.
		for (PackedMap<String, String> sharedProperties : this.sharedPropertiesByFeatures.values()) {
			value = sharedProperties.get(keyKernel);
			if (value != null) {
				try {
					value = Kernel.clone(value, contextOwner);
				} catch (CloneNotSupportedException e) {
					// Cannot occur.
				}
				return value;
			}
		}

		return null;
	}

	@Override
	public void stateChanged(@Nullable Feature feature, @Nullable State previousState) {
		if (feature != null && feature.getState() == State.STOPPED) {
			// Clear properties of the dying feature
			synchronized (this.sharedPropertiesByFeatures) {
				this.sharedPropertiesByFeatures.remove(feature);
			}
		}
	}
}
