/*
 * Copyright 2014-2015 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
/*
 * TERMS OF USE - EASING EQUATIONS
 * Open source under the BSD License.
 *
 * Copyright © 2001 Robert Penner
 * All rights reserved.
 * Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:
 * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer in the documentation and/or other materials provided with the distribution.
 * Neither the name of the author nor the names of contributors may be used to endorse or promote products derived from this software without specific prior written permission.
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE
 */
package ej.motion.bounce;

import ej.motion.AbstractMotion;

/**
 * Represents a bounce motion.
 */
public abstract class AbstractBounceMotion extends AbstractMotion {

	private static final float STEP_FRACTION = 2.75f; // 1 / 2.75 ~ 36%
	private static final float AMPLITUDE = 7.5625f; // Math.pow(2, STEP_FRACTION)
	private static final float FACTOR = 1.5f;
	private static final float INCREMENT_FACTOR = FACTOR / 4;
	private static final float FIRST_LIMIT = 1;
	private static final float SECOND_LIMIT = 2;
	private static final float SECOND_FACTOR = FACTOR;
	private static final float THIRD_LIMIT = 2.5f;
	private static final float THIRD_FACTOR = FACTOR * FACTOR;
	private static final float FOURTH_LIMIT = 2.625f;
	private static final float FOURTH_FACTOR = FOURTH_LIMIT;

	/**
	 * Creates a bounce motion for a move from start to stop.
	 *
	 * @param start
	 *            the start value.
	 * @param stop
	 *            the stop value.
	 * @param maxDuration
	 *            the maximum duration of the motion.
	 */
	public AbstractBounceMotion(int start, int stop, long maxDuration) {
		super(start, stop, maxDuration);
	}

	@Override
	protected boolean isFinished(int value) {
		return false;
	}

	/**
	 * Gets the bounce ease in value for the given time.
	 *
	 * @param currentTime
	 *            the elapsed time.
	 * @return the value for the given time.
	 */
	protected int easeIn(long currentTime) {
		return this.stop - easeOut(this.duration - currentTime);
	}

	/**
	 * Gets the bounce ease out value for the given time.
	 *
	 * @param currentTime
	 *            the elapsed time.
	 * @return the value for the given time.
	 */
	protected int easeOut(long currentTime) {
		float t = (float) currentTime / this.duration;
		float increment;
		if (t < (FIRST_LIMIT / STEP_FRACTION)) {
			increment = 0;
		} else if (t < (SECOND_LIMIT / STEP_FRACTION)) {
			t -= (SECOND_FACTOR / STEP_FRACTION);
			increment = INCREMENT_FACTOR * SECOND_LIMIT;
		} else if (t < (THIRD_LIMIT / STEP_FRACTION)) {
			t -= (THIRD_FACTOR / STEP_FRACTION);
			increment = INCREMENT_FACTOR * THIRD_LIMIT;
		} else {
			t -= (FOURTH_FACTOR / STEP_FRACTION);
			increment = INCREMENT_FACTOR * FOURTH_LIMIT;
		}
		return (int) (this.start + (this.stop - this.start) * (AMPLITUDE * t * t + increment));
	}

}
