/*
 * Copyright (c) 1999, 2012, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package javax.net.ssl;

import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;

import ej.annotation.Nullable;

/**
 * Instances of this class represent a secure socket protocol implementation which acts as a factory for secure socket
 * factories or <code>SSLEngine</code>s. This class is initialized with an optional set of key and trust managers and
 * source of secure random bytes.
 *
 * <p>
 * Every implementation of the Java platform is required to support the following standard <code>SSLContext</code>
 * protocol:
 * <ul>
 * <li><tt>TLSv1</tt></li>
 * </ul>
 * This protocol is described in the <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#SSLContext">
 * SSLContext section</a> of the Java Cryptography Architecture Standard Algorithm Name Documentation. Consult the
 * release documentation for your implementation to see if any other algorithms are supported.
 *
 * @since 1.4
 */
public class SSLContext {

	/**
	 * Returns the default SSL context.
	 *
	 * <p>
	 * The first call of this method triggers the call <code>SSLContext.getInstance("Default")</code>. If successful,
	 * that object is made the default SSL context and returned.
	 *
	 * <p>
	 * The default context is immediately usable and does not require {@linkplain #init initialization}.
	 *
	 * @return the default SSL context
	 * @throws NoSuchAlgorithmException
	 *             if the {@link SSLContext#getInstance SSLContext.getInstance()} call fails
	 * @since 1.6
	 */
	public static synchronized SSLContext getDefault() throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	/**
	 * Returns a <code>SSLContext</code> object that implements the specified secure socket protocol.
	 *
	 * <p>
	 * This method traverses the list of registered security Providers, starting with the most preferred Provider. A new
	 * SSLContext object encapsulating the SSLContextSpi implementation from the first Provider that supports the
	 * specified protocol is returned.
	 *
	 *
	 * @param protocol
	 *            the standard name of the requested protocol. See the SSLContext section in the
	 *            <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#SSLContext"> Java Cryptography
	 *            Architecture Standard Algorithm Name Documentation</a> for information about standard protocol names.
	 *
	 * @return the new <code>SSLContext</code> object.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if no Provider supports a SSLContextSpi implementation for the specified protocol.
	 * @exception NullPointerException
	 *                if protocol is null.
	 *
	 */
	public static SSLContext getInstance(String protocol) throws NoSuchAlgorithmException {
		throw new RuntimeException();
	}

	/**
	 * Returns the protocol name of this <code>SSLContext</code> object.
	 *
	 * <p>
	 * This is the same name that was specified in one of the <code>getInstance</code> calls that created this
	 * <code>SSLContext</code> object.
	 *
	 * @return the protocol name of this <code>SSLContext</code> object.
	 */
	public final String getProtocol() {
		throw new RuntimeException();
	}

	/**
	 * Initializes this context. Either of the first two parameters may be null in which case the installed security
	 * providers will be searched for the highest priority implementation of the appropriate factory. Likewise, the
	 * secure random parameter may be null in which case the default implementation will be used.
	 * <P>
	 * Only the first instance of a particular key and/or trust manager implementation type in the array is used. (For
	 * example, only the first javax.net.ssl.X509KeyManager in the array will be used.)
	 *
	 * @param km
	 *            the sources of authentication keys or null
	 * @param tm
	 *            the sources of peer authentication trust decisions or null
	 * @param random
	 *            the source of randomness for this generator or null
	 * @throws KeyManagementException
	 *             if this operation fails
	 */
	public final void init(@Nullable KeyManager[] km, @Nullable TrustManager[] tm, @Nullable SecureRandom random)
			throws KeyManagementException {
		throw new RuntimeException();
	}

	/**
	 * Returns a <code>SocketFactory</code> object for this context.
	 *
	 * @return the <code>SocketFactory</code> object
	 * @throws IllegalStateException
	 *             if the SSLContextImpl requires initialization and the <code>init()</code> has not been called
	 */
	public final SSLSocketFactory getSocketFactory() {
		throw new RuntimeException();
	}

	/**
	 * Returns a <code>ServerSocketFactory</code> object for this context.
	 *
	 * @return the <code>ServerSocketFactory</code> object
	 * @throws IllegalStateException
	 *             if the SSLContextImpl requires initialization and the <code>init()</code> has not been called
	 */
	public final SSLServerSocketFactory getServerSocketFactory() {
		throw new RuntimeException();
	}

}
