/*
 * Copyright (c) 1998, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright (C) 2015-2020 MicroEJ Corp. EDC compliance and optimizations.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package java.security;

import java.io.IOException;
import java.io.InputStream;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.Enumeration;

import ej.annotation.Nullable;

/**
 * This class defines the <i>Service Provider Interface</i> (<b>SPI</b>) for the {@code KeyStore} class. All the
 * abstract methods in this class must be implemented by each cryptographic service provider who wishes to supply the
 * implementation of a keystore for a particular keystore type.
 *
 * @author Jan Luehe
 *
 *
 * @see KeyStore
 *
 * @since 1.2
 */

public abstract class KeyStoreSpi {

	/**
	 * Returns the key associated with the given alias, using the given password to recover it. The key must have been
	 * associated with the alias by a call to {@code setKeyEntry}, or by a call to {@code setEntry} with a
	 * {@code PrivateKeyEntry} or {@code SecretKeyEntry}.
	 *
	 * @param alias
	 *            the alias name
	 * @param password
	 *            the password for recovering the key
	 *
	 * @return the requested key, or null if the given alias does not exist or does not identify a key-related entry.
	 *
	 * @exception NoSuchAlgorithmException
	 *                if the algorithm for recovering the key cannot be found
	 * @exception UnrecoverableKeyException
	 *                if the key cannot be recovered (e.g., the given password is wrong).
	 */
	@Nullable
	public abstract Key engineGetKey(String alias, char[] password)
			throws NoSuchAlgorithmException, UnrecoverableKeyException;

	/**
	 * Returns the certificate chain associated with the given alias. The certificate chain must have been associated
	 * with the alias by a call to {@code setKeyEntry}, or by a call to {@code setEntry} with a {@code PrivateKeyEntry}.
	 *
	 * @param alias
	 *            the alias name
	 *
	 * @return the certificate chain (ordered with the user's certificate first and the root certificate authority
	 *         last), or null if the given alias does not exist or does not contain a certificate chain
	 */
	@Nullable
	public abstract Certificate[] engineGetCertificateChain(String alias);

	/**
	 * Returns the certificate associated with the given alias.
	 *
	 * <p>
	 * If the given alias name identifies an entry created by a call to {@code setCertificateEntry}, or created by a
	 * call to {@code setEntry} with a {@code TrustedCertificateEntry}, then the trusted certificate contained in that
	 * entry is returned.
	 *
	 * <p>
	 * If the given alias name identifies an entry created by a call to {@code setKeyEntry}, or created by a call to
	 * {@code setEntry} with a {@code PrivateKeyEntry}, then the first element of the certificate chain in that entry
	 * (if a chain exists) is returned.
	 *
	 * @param alias
	 *            the alias name
	 *
	 * @return the certificate, or null if the given alias does not exist or does not contain a certificate.
	 */
	@Nullable
	public abstract Certificate engineGetCertificate(String alias);

	/**
	 * Assigns the given key (that has already been protected) to the given alias.
	 *
	 * <p>
	 * If the protected key is of type {@code java.security.PrivateKey}, it must be accompanied by a certificate chain
	 * certifying the corresponding public key.
	 *
	 * <p>
	 * If the given alias already exists, the keystore information associated with it is overridden by the given key
	 * (and possibly certificate chain).
	 *
	 * @param alias
	 *            the alias name
	 * @param key
	 *            the key (in protected format) to be associated with the alias
	 * @param chain
	 *            the certificate chain for the corresponding public key (only useful if the protected key is of type
	 *            {@code java.security.PrivateKey}).
	 *
	 * @exception KeyStoreException
	 *                if this operation fails.
	 */
	public abstract void engineSetKeyEntry(String alias, byte[] key, Certificate[] chain) throws KeyStoreException;

	/**
	 * Assigns the given certificate to the given alias.
	 *
	 * <p>
	 * If the given alias identifies an existing entry created by a call to {@code setCertificateEntry}, or created by a
	 * call to {@code setEntry} with a {@code TrustedCertificateEntry}, the trusted certificate in the existing entry is
	 * overridden by the given certificate.
	 *
	 * @param alias
	 *            the alias name
	 * @param cert
	 *            the certificate
	 *
	 * @exception KeyStoreException
	 *                if the given alias already exists and does not identify an entry containing a trusted certificate,
	 *                or this operation fails for some other reason.
	 */
	public abstract void engineSetCertificateEntry(String alias, Certificate cert) throws KeyStoreException;

	/**
	 * Deletes the entry identified by the given alias from this keystore.
	 *
	 * @param alias
	 *            the alias name
	 *
	 * @exception KeyStoreException
	 *                if the entry cannot be removed.
	 */
	public abstract void engineDeleteEntry(String alias) throws KeyStoreException;

	/**
	 * Lists all the alias names of this keystore.
	 *
	 * @return enumeration of the alias names
	 */
	public abstract Enumeration<String> engineAliases();

	/**
	 * Checks if the given alias exists in this keystore.
	 *
	 * @param alias
	 *            the alias name
	 *
	 * @return true if the alias exists, false otherwise
	 */
	public abstract boolean engineContainsAlias(String alias);

	/**
	 * Retrieves the number of entries in this keystore.
	 *
	 * @return the number of entries in this keystore
	 */
	public abstract int engineSize();

	/**
	 * Returns true if the entry identified by the given alias was created by a call to {@code setKeyEntry}, or created
	 * by a call to {@code setEntry} with a {@code PrivateKeyEntry} or a {@code SecretKeyEntry}.
	 *
	 * @param alias
	 *            the alias for the keystore entry to be checked
	 *
	 * @return true if the entry identified by the given alias is a key-related, false otherwise.
	 */
	public abstract boolean engineIsKeyEntry(String alias);

	/**
	 * Returns true if the entry identified by the given alias was created by a call to {@code setCertificateEntry}, or
	 * created by a call to {@code setEntry} with a {@code TrustedCertificateEntry}.
	 *
	 * @param alias
	 *            the alias for the keystore entry to be checked
	 *
	 * @return true if the entry identified by the given alias contains a trusted certificate, false otherwise.
	 */
	public abstract boolean engineIsCertificateEntry(String alias);

	/**
	 * Loads the keystore from the given input stream.
	 *
	 * <p>
	 * A password may be given to unlock the keystore (e.g. the keystore resides on a hardware token device), or to
	 * check the integrity of the keystore data. If a password is not given for integrity checking, then integrity
	 * checking is not performed.
	 *
	 * @param stream
	 *            the input stream from which the keystore is loaded, or {@code null}
	 * @param password
	 *            the password used to check the integrity of the keystore, the password used to unlock the keystore, or
	 *            {@code null}
	 *
	 * @exception IOException
	 *                if there is an I/O or format problem with the keystore data, if a password is required but not
	 *                given, or if the given password was incorrect. If the error is due to a wrong password, the
	 *                {@link Throwable#getCause cause} of the {@code IOException} should be an
	 *                {@code UnrecoverableKeyException}
	 * @exception NoSuchAlgorithmException
	 *                if the algorithm used to check the integrity of the keystore cannot be found
	 * @exception CertificateException
	 *                if any of the certificates in the keystore could not be loaded
	 */
	public abstract void engineLoad(@Nullable InputStream stream, @Nullable char[] password)
			throws IOException, NoSuchAlgorithmException, CertificateException;

}
