/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Class for testing the flush limits (see M0092MEJAUI-2388 and M0092MEJAUI-2389).
 */
@SuppressWarnings("nls")
public class TestFontFlushLimits {

	private static final String STRING_SQUARE = "A";

	private static final int PADDING = 1;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string.
	 */
	@Test
	public void testFlushLimitsDrawString() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		int fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check character : full box filled
		int stringWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		TestUtilities.checkArea("flush limits", color, x, y, stringWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("flush limits", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize,
				50, 3);
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string with matrix.
	 */
	@Test
	public void testFlushLimitsDrawStringWithMatrix() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		int fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, matrix, GraphicsContext.OPAQUE, BlendMode.SRC_OVER,
				0);
		display.flush();

		// Check character : full box filled
		int stringWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		TestUtilities.checkArea("flush limits", color, x, y, stringWidth, fontSize, PADDING);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("flush limits", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize,
				50, 3);
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string with gradient.
	 */
	@Test
	public void testFlushLimitsDrawGradientString() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		int fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		int stringWidth = (int) font.measureStringWidth(string.substring(0, 1), fontSize);
		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0, 1 };

		LinearGradient gradient = new LinearGradient(0, 0, stringWidth - 1, 0, colors, stops);

		g.setColor(color);
		VectorGraphicsPainter.drawGradientString(g, string, font, fontSize, matrix, gradient, GraphicsContext.OPAQUE,
				BlendMode.SRC_OVER, 0);
		display.flush();

		// Check character
		TestUtilities.checkNot("flush limits top-left", x, y, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("flush limits bottom-left", x, y + fontSize - 1, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("flush limits top-right", x + stringWidth - 1, y, g, TestUtilities.BACKGROUND_COLOR);
		TestUtilities.checkNot("flush limits bottom-right", x + stringWidth - 1, y + fontSize - 1, g,
				TestUtilities.BACKGROUND_COLOR);

		// Check no drawing outside
		TestUtilities.checkPeripheralArea("flush limits", TestUtilities.BACKGROUND_COLOR, x, y, stringWidth, fontSize,
				50, 3);
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string on circle.
	 */
	@Test
	public void testFlushLimitsDrawStringOnCircle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 48;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		g.setColor(color);
		int radius = 100;
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		float charWidth = font.measureStringWidth(string.substring(0, 1), fontSize);
		// angle offset is negative in clockwise direction
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		float charAngle = -charWidth / (topCircleRadius - fontSize / 2);

		// target the top of the character
		float targetToCenter = topCircleRadius - fontSize / 10;
		int targetX = (int) (x + targetToCenter * Math.cos(charAngle / 2));
		int targetY = (int) (y - targetToCenter * Math.sin(charAngle / 2));
		TestUtilities.check("flush limits top", targetX, targetY, g, color);

		// target the bottom of the character
		targetToCenter = topCircleRadius - fontSize * 9 / 10;
		targetX = (int) (x + targetToCenter * Math.cos(charAngle / 2));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle / 2));
		TestUtilities.check("flush limits bottom", targetX, targetY, g, color);

		// target the left of the character
		targetToCenter = topCircleRadius - fontSize / 2;
		targetX = (int) (x + targetToCenter * Math.cos(charAngle / 10));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle / 10));
		TestUtilities.check("flush limits left", targetX, targetY, g, color);

		// target the right of the character
		targetX = (int) (x + targetToCenter * Math.cos(charAngle * 9 / 10));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle * 9 / 10));
		TestUtilities.check("flush limits right", targetX, targetY, g, color);
	}

	/**
	 * Tests that the flush limits are correctly computed when drawing a string on circle with gradient.
	 */
	@Test
	public void testFlushLimitsDrawGradientStringOnCircle() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		String string = STRING_SQUARE;
		float fontSize = 48;
		int radius = 100;
		int color = Colors.WHITE;
		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;

		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		float stringWidth = font.measureStringWidth(string, fontSize);
		float bottomOfStringToCircleCenter = radius + font.getBaselinePosition(fontSize) - fontSize;
		float centerOfStringToCircleCenter = bottomOfStringToCircleCenter + fontSize / 2;
		// angle offset is negative in clockwise direction
		float stringAngle = -stringWidth / centerOfStringToCircleCenter;

		int[] colors = { TestUtilities.getColorWithAlpha(Colors.BLUE, GraphicsContext.OPAQUE),
				TestUtilities.getColorWithAlpha(Colors.GREEN, GraphicsContext.OPAQUE) };
		float[] stops = { 0.2f, 0.8f };

		float gradientEndX = (float) (bottomOfStringToCircleCenter * Math.cos(stringAngle));
		float gradientEndY = (float) (-bottomOfStringToCircleCenter * Math.sin(stringAngle));

		LinearGradient gradient = new LinearGradient(bottomOfStringToCircleCenter, 0, gradientEndX, gradientEndY,
				colors, stops);

		Painter.drawLine(g, (int) bottomOfStringToCircleCenter, 0, (int) gradientEndX, (int) gradientEndY);
		g.setColor(color);

		VectorGraphicsPainter.drawGradientStringOnCircle(g, string, font, fontSize, matrix, gradient, radius,
				Direction.CLOCKWISE, GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		float charWidth = font.measureStringWidth(string.substring(0, 1), fontSize);
		// angle offset is negative in clockwise direction
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		float charAngle = -charWidth / (topCircleRadius - fontSize / 2);

		// target the top of the character
		float targetToCenter = topCircleRadius - fontSize / 10;
		int targetX = (int) (x + targetToCenter * Math.cos(charAngle / 2));
		int targetY = (int) (y - targetToCenter * Math.sin(charAngle / 2));
		TestUtilities.checkNot("flush limits top", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		// target the bottom of the character
		targetToCenter = topCircleRadius - fontSize * 9 / 10;
		targetX = (int) (x + targetToCenter * Math.cos(charAngle / 2));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle / 2));
		TestUtilities.checkNot("flush limits bottom", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		// target the left of the character
		targetToCenter = topCircleRadius - fontSize / 2;
		targetX = (int) (x + targetToCenter * Math.cos(charAngle / 10));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle / 10));
		TestUtilities.checkNot("flush limits left", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);

		// target the right of the character
		targetX = (int) (x + targetToCenter * Math.cos(charAngle * 9 / 10));
		targetY = (int) (y - targetToCenter * Math.sin(charAngle * 9 / 10));
		TestUtilities.checkNot("flush limits right", targetX, targetY, g, TestUtilities.BACKGROUND_COLOR);
	}
}
