/*
 * Copyright 2010-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */

package ej.sni;

/**
 * This utility class provides methods to convert between Java &amp; C strings.
 * <p>
 * Note: the platform default encoding is used. For converting using a different encoding, see
 * {@link String#String(byte[], String)} &amp; {@link String#getBytes(String)}.
 * <p>
 * Example:
 *
 * <pre>
 * package example;
 *
 * public class Foo {
 *
 * 	private static final int MAX_STRING_SIZE = 42; // including the '\0' character
 *
 * 	public void pushString(String str) {
 * 		pushString(SNI.toCString(str));
 * 	}
 *
 * 	public String pullString() {
 * 		byte[] buffer = new byte[MAX_STRING_SIZE];
 * 		pullString(buffer);
 * 		return SNI.toJavaString(buffer);
 * 	}
 *
 * 	private static native void pushString(byte[] str);
 *
 * 	private static native void pullString(byte[] buffer);
 *
 * }
 * </pre>
 *
 * <pre>
 * #include &lt;sni.h&gt;
 * #include &lt;string.h&gt;
 *
 * #define MAX_STRING_SIZE 42
 *
 * static char gStr[MAX_STRING_SIZE];
 *
 * void Java_example_Foo_pushString(jbyte *str) {
 *     strncpy(gStr, (char*) str, MAX_STRING_SIZE);
 * }
 *
 * void Java_example_Foo_pullString(jbyte *buffer) {
 *     strncpy((char*) buffer, gStr, MAX_STRING_SIZE);
 * }
 * </pre>
 */
public class SNI {

	private SNI() {
		// Forbid instantiation
	}

	/**
	 * Transforms a Java String into a C String.<br>
	 * The platform default encoding is used to transform Java characters into C characters.<br>
	 * The created C String is a NULL terminated String (ends with '\0'). The <code>cString</code> array length must be
	 * at least <code>javaString.length()+1</code>.
	 *
	 * @param javaString
	 *            the Java String to convert.
	 * @param cString
	 *            byte array which contains the C String.
	 *
	 * @throws IllegalArgumentException
	 *             if javaString or cString is null.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if cString is too small to contain the string.
	 */
	public static void toCString(String javaString, byte[] cString) {
		throw new RuntimeException();
	}

	/**
	 * Transforms a Java String into a C String.<br>
	 * The platform default encoding is used to transform Java characters into C characters.<br>
	 * The returned C String is a NULL terminated String (ends with '\0').
	 *
	 * @param javaString
	 *            the Java String to convert.
	 *
	 * @return a new array that contains the C String representation of the given Java String.
	 *
	 * @throws IllegalArgumentException
	 *             if javaString is null.
	 */
	public static byte[] toCString(String javaString) {
		throw new RuntimeException();
	}

	/**
	 * Transforms a C String into a Java String, using platform default encoding. The C String must be NULL terminated.
	 *
	 * @param cString
	 *            byte array which contains the C String.
	 * @return a new Java String.
	 *
	 * @throws IllegalArgumentException
	 *             if cString is null or its length is &lt; 1.
	 * @throws IllegalArgumentException
	 *             if cString is not NULL terminated.
	 */
	public static String toJavaString(byte[] cString) {
		throw new RuntimeException();
	}

}
