/*
 * Java
 *
 * Copyright 2018-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.net.util;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.Enumeration;

import ej.util.message.Level;

/**
 * Utilitary functions about net.
 */
public class NetUtil {

	private NetUtil() {
		// Forbid instantiation
	}

	/**
	 * Checks if a specific network interface is found.
	 *
	 * @param interfaceName
	 *            the interface name.
	 * @return the {@link NetworkInterface}, <code>null</code> if not found.
	 * @throws IOException
	 *             if an IOException occurs.
	 */
	public static NetworkInterface getInterface(String interfaceName) throws IOException {
		NetworkInterface foundInterface = null;
		Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();

		/**
		 * Check if we have a valid enumeration of network interfaces
		 */
		if (interfaces == null) {
			throw new IOException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.ERROR_NO_INTERFACE));
		}

		/**
		 * Look through all of the enumeration and compare with the one to be found
		 */
		while (interfaces.hasMoreElements()) {
			NetworkInterface networkInterface = interfaces.nextElement();
			if (networkInterface.getName().compareTo(interfaceName) == 0) {
				foundInterface = networkInterface;
				break;
			}
		}
		return foundInterface;
	}

	/**
	 * Gets the the first host address of a valid (connected and not loopback) interface. This function does not ensure
	 * to provide the same {@link InetAddress} for two consecutive call if two interfaces are valid.
	 *
	 * @return the host address, <code>null</code> if none found.
	 * @see #getFirstHostAddress(NetworkInterface)
	 */
	public static InetAddress getFirstHostAddress() {
		InetAddress ip = null;
		try {
			Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
			if (interfaces != null) {
				while (interfaces.hasMoreElements()) {
					NetworkInterface iface = interfaces.nextElement();
					ip = getFirstHostAddress(iface);
					if (ip != null) {
						break;
					}
				}
			}
		} catch (SocketException e) {
			// An issue append, the board is not connected.
			Messages.LOGGER.log(Level.SEVERE, Messages.CATEGORY, Messages.ERROR_COULD_NOT_GET_IP, e);
		}
		return ip;
	}

	/**
	 * Gets the first host address of a valid (connected and not loopback) interface.
	 *
	 * @param iface
	 *            the interface to gets the host address from, can be <code>null</code>.
	 *
	 * @return the host address, <code>null</code> if none found.
	 */
	public static InetAddress getFirstHostAddress(NetworkInterface iface) {
		InetAddress ip = null;
		try {
			// filters out 127.0.0.1 and inactive interfaces
			if (iface != null && !iface.isLoopback() && iface.isUp()) {
				Enumeration<InetAddress> addresses = iface.getInetAddresses();
				while (addresses.hasMoreElements()) {
					InetAddress addr = addresses.nextElement();
					if (isValidInetAdress(addr)) {
						ip = addr;
						break;
					}
				}
			}

		} catch (SocketException e) {
			// An issue append, the board is not connected.
			Messages.LOGGER.log(Level.SEVERE, Messages.CATEGORY, Messages.ERROR_COULD_NOT_GET_IP, e);
		}
		return ip;
	}

	/**
	 * Checks whether an IP is not loopback nor zero.
	 *
	 * @param addr
	 *            the {@link InetAddress} address to check, can be <code>null</code>.
	 * @return <code>true</code> if the IP is valid.
	 */
	public static boolean isValidInetAdress(InetAddress addr) {
		return addr != null && !addr.isLoopbackAddress() && !isConnected(addr);
	}

	/**
	 * Checks whether an {@link InetAddress} is connected to a network.
	 *
	 * @param address
	 *            the address to check, can be <code>null</code>.
	 * @return <code>true</code> if the address has been initialized.
	 */
	public static boolean isConnected(InetAddress address) {
		boolean isConnected = false;

		if (address != null) {
			byte[] bs = address.getAddress();
			isConnected = true;
			for (int i = 0; i < bs.length; i++) {
				if (bs[i] != 0) {
					isConnected = false;
					break;
				}
			}
		}
		return isConnected;
	}

}
