/*
 * Java
 *
 * Copyright 2009-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.fp.widget;

import ej.fp.MouseListener;
import ej.fp.Widget.WidgetAttribute;
import ej.fp.Widget.WidgetDescription;
import ej.fp.util.WidgetWithListener;
import ej.microui.event.EventPointer;
import ej.microui.event.EventTouch;

/**
 * Widget to simulate a mouse or a touch panel. It manages too the states for pointer buttons <i>press</i> and
 * <i>release</i>.
 * <p>
 * Notes:
 * <ul>
 * <li>This widget requires a listener: an implementation of {@link PointerListener} to send the pointer events,</li>
 * <li>When {@link #setTouch(boolean)} attribute is false, this widget manages a mouse. Until three buttons are
 * managed,</li>
 * <li>When {@link #setTouch(boolean)} attribute is true, this widget manages a touch panel. Only one button is
 * managed.</li>
 * </ul>
 */
@SuppressWarnings("nls")
@WidgetDescription(attributes = { @WidgetAttribute(name = "label", isOptional = true), @WidgetAttribute(name = "x"),
		@WidgetAttribute(name = "y"), @WidgetAttribute(name = "width"), @WidgetAttribute(name = "height"),
		@WidgetAttribute(name = "filter", isOptional = true), @WidgetAttribute(name = "areaWidth", isOptional = true),
		@WidgetAttribute(name = "areaHeight", isOptional = true), @WidgetAttribute(name = "touch", isOptional = true),
		@WidgetAttribute(name = "listenerClass", isOptional = true) })
public class Pointer extends WidgetWithListener implements MouseListener {

	/**
	 * Defines the delay (in milliseconds) the system has to respect to send a new move event to the application. Under
	 * this delay, the move event is filtered and not sent to the application.
	 * <p>
	 * The default delay is 20ms which is a standard delay in embedded touch drivers.
	 */
	private static final int DELAY_MOVE_EVENTS_MS = 20;

	/**
	 * Interface that handle pointer events.
	 */
	public static interface PointerListener {

		/**
		 * The specified pointer button has been pressed at given location.
		 *
		 * @param widget
		 *            the pointer that has been updated.
		 * @param x
		 *            the x coordinate of the press position.
		 * @param y
		 *            the y coordinate of the press position.
		 * @param button
		 *            the button that has been pressed.
		 */
		void press(Pointer widget, int x, int y, MouseButton button);

		/**
		 * The specified pointer button has been released.
		 *
		 * @param widget
		 *            the pointer that has been updated.
		 * @param button
		 *            the button that has been released.
		 */
		void release(Pointer widget, MouseButton button);

		/**
		 * The pointer has moved to a new position.
		 *
		 * @param widget
		 *            the pointer that has been updated.
		 * @param x
		 *            the x coordinate of the new position.
		 * @param y
		 *            the y coordinate of the new position.
		 */
		void move(Pointer widget, int x, int y);
	}

	/**
	 * Default implementation of {@link PointerListener}.
	 * <p>
	 * This implementation sends some MicroUI Pointer events using the button ID. It targets the common MicroUI Pointer
	 * generator identified by the tag {@link EventPointer#COMMON_MICROUI_GENERATOR_TAG} or
	 * {@link EventTouch#COMMON_MICROUI_GENERATOR_TAG}.
	 * <p>
	 * This tag can be changed overriding the method {@link #getMicroUIGeneratorTag(Pointer)}.
	 */
	public static class PointerListenerToPointerEvents implements PointerListener {

		@Override
		public void move(Pointer widget, int x, int y) {
			EventTouch.sendMovedEvent(getMicroUIGeneratorTag(widget), x, y);
		}

		@Override
		public void press(Pointer widget, int x, int y, MouseButton button) {
			EventTouch.sendPressedEvent(getMicroUIGeneratorTag(widget), x, y);
		}

		@Override
		public void release(Pointer widget, MouseButton button) {
			EventTouch.sendReleasedEvent(getMicroUIGeneratorTag(widget));
		}

		/**
		 * Gets the MicroUI Pointer events generator tag. This generator has to match the generator set during the
		 * MicroEJ platform build in <code>microui/microui.xml</code>.
		 *
		 * @param widget
		 *            the pointer that has been updated.
		 *
		 * @return a MicroUI Pointer events generator tag.
		 */
		protected String getMicroUIGeneratorTag(Pointer widget) {
			return widget.isTouch() ? EventTouch.COMMON_MICROUI_GENERATOR_TAG
					: EventPointer.COMMON_MICROUI_GENERATOR_TAG;
		}
	}

	private PointerListener listener;

	// false or zero init
	private long lastEventTime;
	private boolean touch;
	private boolean drag;
	private int areaWidth;
	private int areaHeight;

	/**
	 * Sets the area width of the pointer area. If not set, the area width is equals to the widget width (see
	 * {@link #finalizeConfiguration()}).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param width
	 *            the width to set.
	 */
	public void setAreaWidth(int width) {
		if (width < 0) {
			throw new IllegalArgumentException("Area width cannot be negative.");
		}
		this.areaWidth = width;
	}

	/**
	 * Sets the area height of the pointer area. If not set, the area height is equals to the widget height (see
	 * {@link #finalizeConfiguration()}).
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param height
	 *            the height to set.
	 */
	public void setAreaHeight(int height) {
		if (height < 0) {
			throw new IllegalArgumentException("Area height cannot be negative.");
		}
		this.areaHeight = height;
	}

	/**
	 * Configures touch or mouse mode.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param touch
	 *            true to set touch mode.
	 */
	public void setTouch(boolean touch) {
		this.touch = touch;
	}

	/**
	 * Defines the user class which has to implement {@link PointerListener}.
	 * <p>
	 * This method should only be called by front panel parser.
	 *
	 * @param listenerClassName
	 *            user listener class name.
	 */
	public void setListenerClass(String listenerClassName) {
		setListenerClass(PointerListener.class, listenerClassName);
	}

	@Override
	public synchronized void finalizeConfiguration() {
		super.finalizeConfiguration();

		// size for application may be different than size on front panel
		if (this.areaWidth == 0) {
			this.areaWidth = getWidth();
		}
		if (this.areaHeight == 0) {
			this.areaHeight = getHeight();
		}
	}

	@Override
	public void start() {
		super.start();

		// retrieve the listener
		this.listener = newListener(PointerListener.class);
	}

	@Override
	protected Object newDefaultListener() {
		return new PointerListenerToPointerEvents();
	}

	@Override
	public void mouseEntered(int x, int y) {
		mouseMoved(x, y);
	}

	@Override
	public void mouseExited(int x, int y) {
		mouseMoved(x, y);
	}

	@Override
	public void mouseMoved(int x, int y) {
		if (!this.touch) {
			mouseMove(x, y);
		}
	}

	@Override
	public void mouseDragged(int x, int y) {
		if (!this.touch || this.drag) {
			mouseMove(x, y);
		}
	}

	@Override
	public void mousePressed(int x, int y, MouseButton button) {
		if (!this.touch || button == MouseButton.FIRST_BUTTON) {
			mousePress(x, y, button);
			this.drag = true;
		}
	}

	@Override
	public void mouseReleased(int x, int y, MouseButton button) {
		if (!this.touch || button == MouseButton.FIRST_BUTTON) {
			mouseRelease(x, y, button);
			this.drag = false;
		}
	}

	/**
	 * Gets the widget listener.
	 *
	 * @return the listener which manages the events.
	 */
	protected PointerListener getListener() {
		return this.listener;
	}

	/**
	 * Tells if pointer is a touch panel or a mouse.
	 *
	 * @return true when pointer manages a touch panel.
	 */
	protected boolean isTouch() {
		return this.touch;
	}

	/**
	 * Gets the minimum delay (in milliseconds) the system has to respect to send a new move event to the application.
	 * Under this delay, the move event is filtered and not sent to the application. This filtering allows to reduce the
	 * number of move events sent to the application in order to reduce the application actions (repaint etc.).
	 *
	 * @return minimum delay between two move events.
	 */
	protected int getMoveFilteringDelay() {
		return DELAY_MOVE_EVENTS_MS;
	}

	/**
	 * Computes X coordinate on the screen
	 *
	 * @param x
	 *            X coordinate to compute
	 * @return X coordinate computed
	 */
	private int updatePointerX(int x) {
		x *= this.areaWidth;
		x /= getWidth();
		return x;
	}

	/**
	 * Computes Y coordinate on the screen
	 *
	 * @param x
	 *            Y coordinate to compute
	 * @return Y coordinate computed
	 */
	private int updatePointerY(int y) {
		y *= this.areaHeight;
		y /= getHeight();
		return y;
	}

	private void mouseMove(int x, int y) {
		if (System.currentTimeMillis() - this.lastEventTime > getMoveFilteringDelay()) {
			this.listener.move(this, updatePointerX(x), updatePointerY(y));
			this.lastEventTime = System.currentTimeMillis();
		}
	}

	private void mousePress(int x, int y, MouseButton button) {
		this.listener.press(this, updatePointerX(x), updatePointerY(y), button);
		this.lastEventTime = System.currentTimeMillis();
	}

	private void mouseRelease(int x, int y, MouseButton button) {
		this.listener.release(this, button);
	}
}
